/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Date;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodec;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;

/**
 * Poll message used to indicate the result of a pending action. Pending action
 * can included domain create, update, renew, transfer, and delete.
 *
 * @see com.verisign.epp.codec.domain.EPPDomainCreateCmd
 * @see com.verisign.epp.codec.domain.EPPDomainUpdateCmd
 * @see com.verisign.epp.codec.domain.EPPDomainRenewCmd
 * @see com.verisign.epp.codec.domain.EPPDomainTransferCmd
 * @see com.verisign.epp.codec.domain.EPPDomainDeleteCmd
 */
public class EPPDomainPendActionMsg extends EPPResponse {
	/**
	 * XML local name for {@code EPPDomainPendActionMsg}.
	 */
	public static final String ELM_LOCALNAME = "panData";

	/**
	 * XML Element Name of {@code EPPDomainPendActionMsg} root element.
	 */
	public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name for the {@code name} attribute.
	 */
	private final static String ELM_DOMAIN_NAME = "name";

	/**
	 * XML Element Name for the {@code paTRID} attribute.
	 */
	private final static String ELM_PATRID = "paTRID";

	/**
	 * XML Element Name for the {@code paDate} attribute.
	 */
	private final static String ELM_PENDING_DATE = "paDate";

	/**
	 * XML Attribute Name for the pending action result
	 */
	private final static String ATTR_RESULT = "paResult";

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPDomainPendActionMsg.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * fully qualified domain name of pending action.
	 */
	private String name = null;

	/**
	 * Was the pending request successful? A value of {@code true} indicates that
	 * the request has been approved and completed. A value of {@code false}
	 * indicates that the request has been denied and the requested action has
	 * not been taken. Default value is {@code false}.
	 */
	private boolean paSuccess = false;

	/**
	 * Pending transaction id contains the client transaction identifier and
	 * server transaction identifier returned with the original response to
	 * process the command.
	 */
	EPPTransId pendingTransId = null;

	/**
	 * Pending date and time describing when review of the requested action was
	 * completed.
	 */
	private Date pendingDate = null;

	/**
	 * {@code EPPDomainPendActionMsg} default constructor. Must call required
	 * setter methods before encode.
	 */
	public EPPDomainPendActionMsg() {
	}

	/**
	 * {@code EPPDomainPendActionMsg} constructor that takes all of the required
	 * attributes.
	 *
	 * @param aTransId
	 *           Poll command transaction id
	 * @param aName
	 *           Domain name of pending action
	 * @param aSuccess
	 *           Was the pending action successful
	 * @param aPendingTransId
	 *           Pending action transaction id
	 * @param aPendingDate
	 *           Date of pending action completion
	 */
	public EPPDomainPendActionMsg(EPPTransId aTransId, String aName, boolean aSuccess, EPPTransId aPendingTransId,
	      Date aPendingDate) {
		super(aTransId);

		this.name = aName;
		this.paSuccess = aSuccess;
		this.setPendingTransId(aPendingTransId);
		this.pendingDate = aPendingDate;
	}

	/**
	 * Gets the EPP command type associated with {@code EPPDomainPendActionMsg}.
	 *
	 * @return EPPDomainPendActionMsg.ELM_NAME
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * {@code EPPDomainPendActionMsg}.
	 *
	 * @return {@code EPPDomainMapFactory.NS}
	 */
	public String getNamespace() {
		return EPPDomainMapFactory.NS;
	}

	/**
	 * Gets the domain name
	 *
	 * @return Domain Name {@code String} instance if defined; {@code null}
	 *         otherwise.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the domain name.
	 *
	 * @param aName
	 *           Domain Name
	 */
	public void setName(String aName) {
		name = aName;
	}

	/**
	 * Gets the date that the pending action was completed.
	 *
	 * @return Returns the pendingDate.
	 */
	public Date getPendingDate() {
		return pendingDate;
	}

	/**
	 * Sets the date that the pending action was completed.
	 *
	 * @param pendingDate
	 *           The pendingDate to set.
	 */
	public void setPendingDate(Date pendingDate) {
		this.pendingDate = pendingDate;
	}

	/**
	 * Gets the pending transaction id.
	 *
	 * @return Returns the pendingTransId.
	 */
	public EPPTransId getPendingTransId() {
		return pendingTransId;
	}

	/**
	 * Sets the pending transaction id.
	 *
	 * @param pendingTransId
	 *           The pendingTransId to set.
	 */
	public void setPendingTransId(EPPTransId pendingTransId) {
		this.pendingTransId = pendingTransId;
		this.pendingTransId.setRootName(EPPDomainMapFactory.NS, EPPDomainMapFactory.NS_PREFIX + ":" + ELM_PATRID);
	}

	/**
	 * Was the pending action successful?
	 *
	 * @return Returns {@code true} if the pending action was successfully
	 *         completed; {@code false} otherwise.
	 */
	public boolean isPASuccess() {
		return paSuccess;
	}

	/**
	 * Sets if the pending action was successful.
	 *
	 * @param success
	 *           The success to set.
	 */
	public void setPASuccess(boolean success) {
		this.paSuccess = success;
	}

	/**
	 * Compare an instance of {@code EPPDomainPendActionMsg} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return DOCUMENT ME!
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPDomainPendActionMsg)) {
			cat.error("EPPDomainPendActionMsg.equals(): " + aObject.getClass().getName()
			      + " not EPPDomainPendActionMsg instance");

			return false;
		}

		if (!super.equals(aObject)) {
			cat.error("EPPDomainPendActionMsg.equals(): super class not equal");

			return false;
		}

		EPPDomainPendActionMsg theComp = (EPPDomainPendActionMsg) aObject;

		// Name
		if (!((this.name == null) ? (theComp.name == null) : this.name.equals(theComp.name))) {
			cat.error("EPPDomainPendActionMsg.equals(): name not equal");

			return false;
		}

		// PA Success
		if (this.paSuccess != theComp.paSuccess) {
			cat.error("EPPDomainPendActionMsg.equals(): paSuccess not equal");

			return false;
		}

		// Pending Transaction Id
		if (!((this.pendingTransId == null) ? (theComp.pendingTransId == null) :
		      this.pendingTransId.equals(theComp.pendingTransId))) {
			cat.error("EPPDomainPendActionMsg.equals(): pendingTransId not equal");

			return false;
		}

		// Pending Date
		if (!((this.pendingDate == null) ? (theComp.pendingDate == null) :
		      this.pendingDate.equals(theComp.pendingDate))) {
			cat.error("EPPDomainPendActionMsg.equals(): pendingDate not equal");

			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPDomainPendActionMsg}.
	 *
	 * @return clone of {@code EPPDomainPendActionMsg}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPDomainPendActionMsg clone = (EPPDomainPendActionMsg) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Validate the state of the {@code EPPDomainPendActionMsg} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            DOCUMENT ME!
	 */
	void validateState() throws EPPCodecException {
		if (this.name == null) {
			cat.error("EPPDomainPendActionMsg.validateState(): required attribute name is not set");
			throw new EPPCodecException("required attribute name is not set");
		}

		if (this.pendingTransId == null) {
			cat.error("EPPDomainPendActionMsg.validateState(): required attribute pendingTransId is not set");
			throw new EPPCodecException("required attribute pendingTransId is not set");
		}

		if (this.pendingDate == null) {
			cat.error("EPPDomainPendActionMsg.validateState(): required attribute pendingDate is not set");
			throw new EPPCodecException("required attribute pendingDate is not set");
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPDomainPendActionMsg} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPDomainPendActionMsg} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPDomainPendActionMsg} instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		// Validate Attributes
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPDomainPendActionMsg.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

		// Domain Name
		Element nameElm = aDocument.createElementNS(EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);
		root.appendChild(nameElm);

		// Available
		if (this.paSuccess) {
			nameElm.setAttribute(ATTR_RESULT, "1");
		}
		else {
			nameElm.setAttribute(ATTR_RESULT, "0");
		}

		Text textNode = aDocument.createTextNode(this.name);
		nameElm.appendChild(textNode);

		// Pending Transaction Id
		EPPUtil.encodeComp(aDocument, root, this.pendingTransId);

		// Pending Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.pendingDate, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_PENDING_DATE);

		return root;
	}

	/**
	 * Decode the {@code EPPDomainPendActionMsg} attributes from the
	 * {@code aElement} DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPDomainPendActionMsg} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode {@code aElement}
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {
		Element theNameElm = EPPUtil.getElementByTagNameNS(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);
		this.name = theNameElm.getFirstChild().getNodeValue();

		String resultStr = theNameElm.getAttribute(ATTR_RESULT);

		// paSuccess?
		if (resultStr.equals("1")) {
			this.paSuccess = true;
		}
		else {
			this.paSuccess = false;
		}

		// Pending Transaction ID
		this.pendingTransId = (EPPTransId) EPPUtil.decodeComp(aElement, EPPDomainMapFactory.NS, ELM_PATRID,
		      EPPTransId.class);

		// Pending Date
		this.pendingDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_PENDING_DATE);

		// Validate Attributes
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPDecodeException("Invalid state on EPPDomainPendActionMsg.decode: " + e);
		}
	}

}
