/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryReservedNames} defines the set of reserved domain names
 * starting from that label level. The reserved names can refer to values with
 * more than one level which is relative to the level of the parent
 * &lt;registry:domainName&gt; element.
 */
public class EPPRegistryReservedNames implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryReservedNames.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryReservedNames}.
	 */
	public static final String ELM_LOCALNAME = "reservedNames";

	/**
	 * XML root tag for {@code EPPRegistryReservedNames}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code reservedNames} attribute.
	 */
	final static String ELM_RESERVED_NAME = "reservedName";

	/**
	 * XML Element Name of {@code reservedNameURI} attribute.
	 */
	final static String ELM_RESERVED_NAME_URI = "reservedNameURI";

	/**
	 * Zero or more &lt;registry:reservedName&gt; elements containing a reserved
	 * domain name relative to the level of the parent
	 * &lt;registry:domainName&gt; element.
	 */
	private List<String> reservedNames = new ArrayList<String>();

	/**
	 * An OPTIONAL URI that to an externally defined list of reserved domain name
	 * relative to the level of the parent &lt;registry:domainName&gt; element.
	 */
	private String reservedNameURI = null;

	/**
	 * Default constructor.
	 */
	public EPPRegistryReservedNames() {
	}

	/**
	 * {@code EPPRegistryReservedNames} constructor that takes the list of
	 * reserved names.
	 * 
	 * @param aReservedNames
	 *           List of reserved names relative to the level of the zone.
	 */
	public EPPRegistryReservedNames(List<String> aReservedNames) {
		this.setReservedNames(aReservedNames);
	}

	/**
	 * {@code EPPRegistryReservedNames} constructor that takes the external
	 * reserved name URI.
	 * 
	 * @param aReservedNameURI
	 *           URI to an externally defined list of reserved domain names
	 *           relative to the level of the zone. Set to {@code null} to clear
	 *           the URI.
	 */
	public EPPRegistryReservedNames(String aReservedNameURI) {
		this.reservedNameURI = aReservedNameURI;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryReservedNames} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryReservedNames} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryReservedNames}
	 *               instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryReservedNames.encode: " + e);
		}
		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Reserved Names
		EPPUtil.encodeList(aDocument, root, this.reservedNames, EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_RESERVED_NAME);

		// Reserved Name URI
		EPPUtil.encodeString(aDocument, root, this.reservedNameURI, EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_RESERVED_NAME_URI);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryReservedNames} attributes from the aElement
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryReservedNames}
	 *           from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Reserved Names
		this.reservedNames = EPPUtil.decodeList(aElement, EPPRegistryMapFactory.NS, ELM_RESERVED_NAME);

		// Reserved Name URI
		this.reservedNameURI = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_RESERVED_NAME_URI);
	}

	/**
	 * implements a deep {@code EPPRegistryReservedNames} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryReservedNames} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryReservedNames)) {
			return false;
		}

		EPPRegistryReservedNames theComp = (EPPRegistryReservedNames) aObject;

		// Reserved Names
		if (!((this.reservedNames == null) ? (theComp.reservedNames == null)
		      : EPPUtil.equalLists(this.reservedNames, theComp.reservedNames))) {
			cat.error("EPPRegistryReservedNames.equals(): reservedNames not equal");
			return false;
		}

		// Reserved Name URI
		if (!EqualityUtil.equals(this.reservedNameURI, theComp.reservedNameURI)) {
			cat.error("EPPRegistryReservedNames.equals(): reservedNameURI not equal");
			return false;
		}

		return true;
	}

	/**
	 * Validate the state of the {@code EPPRegistryReservedNames} instance. A
	 * valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	void validateState() throws EPPCodecException {
		if (this.reservedNames != null && this.reservedNames.size() > 0 && this.reservedNameURI != null
		      && this.reservedNameURI.trim().length() > 0) {
			throw new EPPCodecException("reservedNames and reservedNameURI cannot be used together");
		}
	}

	/**
	 * Clone {@code EPPRegistryZoneInfo}.
	 *
	 * @return clone of {@code EPPRegistryZoneInfo}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryReservedNames clone = (EPPRegistryReservedNames) super.clone();
		if (this.reservedNames != null) {
			clone.reservedNames = (List) ((ArrayList) this.reservedNames).clone();
		}
		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Is the reserved name list defined?
	 *
	 * @return {@code true} if the reserved name list is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasReservedNames() {
		return (this.reservedNames != null && !this.reservedNames.isEmpty() ? true : false);
	}

	/**
	 * Gets the reserved names.
	 * 
	 * @return Reserved names if defined; {@code null} otherwise.
	 */
	public List<String> getReservedNames() {
		return this.reservedNames;
	}

	/**
	 * Sets the reserved names.
	 * 
	 * @param aReservedNames
	 *           List of reserved names relative to the zone.
	 */
	public void setReservedNames(List<String> aReservedNames) {
		if (aReservedNames == null) {
			this.reservedNames = new ArrayList<String>();
		}
		else {
			this.reservedNames = aReservedNames;
		}
	}

	/**
	 * Add a reserved name relative to the zone to the list of reserved names.
	 * 
	 * @param aReservedName
	 *           Reserved name to add to the list of reserved names.
	 */
	public void addReservedName(String aReservedName) {
		if (this.reservedNames == null) {
			this.reservedNames = new ArrayList<String>();
		}
		this.reservedNames.add(aReservedName);
	}

	/**
	 * Is the reserved name URI defined?
	 *
	 * @return {@code true} if the reserved name URI is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasReservedNameURI() {
		return (this.reservedNameURI != null ? true : false);
	}

	/**
	 * Gets the URI to an externally defined list of reserved domain names
	 * relative to the level of the zone.
	 * 
	 * @return URI to an externally defined list of reserved domain names
	 *         relative to the level of the zone if defined; {@code null}
	 *         otherwise.
	 */
	public String getReservedNameURI() {
		return this.reservedNameURI;
	}

	/**
	 * Sets the URI to an externally defined list of reserved domain names
	 * relative to the level of the zone.
	 * 
	 * @param aReservedNameURI
	 *           URI to an externally defined list of reserved domain names
	 *           relative to the level of the zone. Set to {@code null} to clear
	 *           the URI.
	 */
	public void setReservedNameURI(String aReservedNameURI) {
		this.reservedNameURI = aReservedNameURI;
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
