/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Defines the policies for external hosts. The &lt;registry:external&gt;
 * elements contains the following child elements: <br>
 * <ul>
 * <li>&lt;registry:minIP&gt; - Minimum number of IP addresses supported for an
 * external host.</li>
 * <li>&lt;registry:maxIP&gt; - Maximum number of IP addresses supported for an
 * external host.</li>
 * <li>&lt;registry:sharePolicy&gt; - The OPTIONAL policy for the sharing of
 * external hosts in the server.The possible shared policy values include:
 * <ul>
 * <li>"perRegistrar" - The external hosts are shared across all domains of the
 * registrar. There is a single pool of external hosts defined per registrar.
 * </li>
 * <li>"perZone" - The external hosts are shared across all domains of the zone.
 * There is a single pool of external hosts defined for the zone.</li>
 * <li>"perSystem" - The external hosts are shared across all zones of the
 * system. There is a single pool of external hosts across all of the zones
 * supported by the system.</li>
 * </ul>
 * </li>
 * <li>&lt;registry:uniqueIpAddressRequired&gt; - The OPTIONAL boolean value
 * that indicates that all of the IP addresses for the host object must be
 * unique, with a default value of "false".</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryHost
 */
public class EPPRegistryExternalHost extends EPPRegistryAbstractMinMax {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryExternalHost.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryExternalHost}.
	 */
	public static final String ELM_LOCALNAME = "external";

	/**
	 * XML root tag for {@code EPPRegistryExternalHost}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code min} attribute.
	 */
	protected static final String ELM_MIN_IP = "minIP";

	/**
	 * XML Element Name of {@code max} attribute.
	 */
	protected static final String ELM_MAX_IP = "maxIP";

	/**
	 * XML Element Name of {@code sharePolicy} attribute.
	 */
	public static final String ELM_SHARE_POLICY = "sharePolicy";

	/**
	 * XML Element Name of {@code uniqueIpAddressesRequired} attribute.
	 */
	public static final String ELM_UNIQUE_IP_ADDRESSES_REQUIRED = "uniqueIpAddressesRequired";

	/**
	 * Possible values for the {@code sharePolicy} attribute.
	 */
	public static enum SharePolicy {

		/**
		 * Constant for "perRegistrar" policy, where the external hosts are shared
		 * across all domains of the registrar. There is a single pool of external
		 * hosts defined per registrar.
		 */
		perRegistrar,

		/**
		 * Constant for "perZone" policy, where the external hosts are shared
		 * across all domains of the zone. There is a single pool of external
		 * hosts defined for the zone.
		 */
		perZone,

		/**
		 * Constant for "perSystem" policy, where external hosts are shared across
		 * all zones of the system. There is a single pool of external hosts
		 * across all of the zones supported by the system.
		 */
		perSystem
	}

	/**
	 * Share policy attribute
	 */
	private SharePolicy sharePolicy = null;

	/**
	 * The OPTIONAL boolean value that indicates that all of the IP addresses for
	 * the host object must be unique, with a default value of "false".
	 */
	private Boolean uniqueIpAddressesRequired = Boolean.FALSE;

	/**
	 * Creates a new {@code EPPRegistryExternalHost} object.
	 */
	public EPPRegistryExternalHost() {
	}

	/**
	 * Constructor that takes {@code min}, {@code max}, {@code sharePolicy}, and
	 * {@code uniqueIpAddressesRequired} using {@code Integer}'s.
	 *
	 * @param aMin
	 *           minimum number of IPs supported for an external host
	 * @param aMax
	 *           maximum number of IPs supported for an external host
	 * @param aSharePolicy
	 *           Share policy to set. Set to {@code null} if undefined.
	 * @param aUniqueIpAddressesRequired
	 *           Indicates that all of the IP addresses for the host object must
	 *           be unique, with a default value of "false". Set to {@code null}
	 *           to set the default.
	 */
	public EPPRegistryExternalHost(Integer aMin, Integer aMax, SharePolicy aSharePolicy,
	      Boolean aUniqueIpAddressesRequired) {
		super(aMin, aMax);

		this.setSharePolicy(aSharePolicy);
		this.setUniqueIpAddressesRequired(aUniqueIpAddressesRequired);
	}

	/**
	 * Constructor that takes {@code min}, {@code max}, {@code sharePolicy}, and
	 * {@code uniqueIpAddressesRequired} using {@code int}'s.
	 *
	 * @param aMin
	 *           minimum number of IPs supported for an external host
	 * @param aMax
	 *           maximum number of IPs supported for an external host
	 * @param aSharePolicy
	 *           Share policy to set. Set to {@code null} if undefined.
	 * @param aUniqueIpAddressesRequired
	 *           Indicates that all of the IP addresses for the host object must
	 *           be unique, with a default value of "false". Set to {@code null}
	 *           to set the default.
	 */
	public EPPRegistryExternalHost(int aMin, int aMax, SharePolicy aSharePolicy, Boolean aUniqueIpAddressesRequired) {
		this(Integer.valueOf(aMin), Integer.valueOf(aMax), aSharePolicy, aUniqueIpAddressesRequired);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryExternalHost} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryExternalHost} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryExternalHost} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		this.validateState();

		Element root = super.encode(aDocument);

		// Share Policy
		if (this.hasSharePolicy()) {
			EPPUtil.encodeString(aDocument, root, this.sharePolicy.toString(), EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_SHARE_POLICY);
		}

		// Unique IP Addresses Required
		EPPUtil.encodeString(aDocument, root, this.uniqueIpAddressesRequired.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_UNIQUE_IP_ADDRESSES_REQUIRED);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryExternalHost} attributes from the aElement
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryExternalHost} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		super.decode(aElement);

		// Share Policy
		String theSharePolicyStr = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_SHARE_POLICY);
		if (theSharePolicyStr == null) {
			this.sharePolicy = null;
		}
		else {
			this.sharePolicy = SharePolicy.valueOf(theSharePolicyStr);
		}

		// Unique IP Addresses Required
		this.setUniqueIpAddressesRequired(
		      EPPUtil.decodeBoolean(aElement, EPPRegistryMapFactory.NS, ELM_UNIQUE_IP_ADDRESSES_REQUIRED));

	}

	/**
	 * Clone {@code EPPRegistryExternalHost}.
	 *
	 * @return clone of {@code EPPRegistryExternalHost}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryExternalHost} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryExternalHost} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!super.equals(aObject)) {
			return false;
		}

		if (!(aObject instanceof EPPRegistryExternalHost)) {
			return false;
		}

		EPPRegistryExternalHost theComp = (EPPRegistryExternalHost) aObject;

		// Share Policy
		if (!EqualityUtil.equals(this.sharePolicy, theComp.sharePolicy)) {
			getLogger().error("EPPRegistryExternalHost.equals(): sharePolicy not equal");
			return false;
		}

		// Unique IP Addresses Required
		if (!EqualityUtil.equals(this.uniqueIpAddressesRequired, theComp.uniqueIpAddressesRequired)) {
			getLogger().error("EPPRegistryExternalHost.equals(): uniqueIpAddressesRequired not equal");
			return false;
		}

		return true;
	}

	/**
	 * Is the share policy defined?
	 *
	 * @return {@code true} if the share policy is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasSharePolicy() {
		return (this.sharePolicy != null ? true : false);
	}

	/**
	 * Gets the share policy.
	 *
	 * @return share policy if defined; {@code null} otherwise.
	 */
	public SharePolicy getSharePolicy() {
		return this.sharePolicy;
	}

	/**
	 * Sets the share policy.
	 *
	 * @param aSharePolicy
	 *           Share policy. Set to {@code null} to clear it.
	 */
	public void setSharePolicy(SharePolicy aSharePolicy) {
		this.sharePolicy = aSharePolicy;
	}

	/**
	 * Gets the OPTIONAL boolean value that indicates that all of the IP
	 * addresses for the host object must be unique, with a default value of
	 * "false".
	 *
	 * @return the {@code uniqueIpAddressesRequired} boolean value.
	 */
	public Boolean getUniqueIpAddressesRequired() {
		return this.uniqueIpAddressesRequired;
	}

	/**
	 * Sets the OPTIONAL boolean value that indicates that all of the IP
	 * addresses for the host object must be unique, with a default value of
	 * "false".
	 *
	 * @param aUniqueIpAddressesRequired
	 *           Set the boolean value. If set to {@code null} the default value
	 *           of "false" will be set.
	 */
	public void setUniqueIpAddressesRequired(Boolean aUniqueIpAddressesRequired) {
		if (aUniqueIpAddressesRequired == null) {
			this.uniqueIpAddressesRequired = Boolean.FALSE;
		}
		else {
			this.uniqueIpAddressesRequired = aUniqueIpAddressesRequired;
		}
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the root element local name.
	 *
	 * @return Root element local name.
	 */
	@Override
	protected String getRootName() {
		return ELM_LOCALNAME;
	}

	/**
	 * Gets the minimum element local name.
	 *
	 * @return Minimum element local name.
	 */
	@Override
	protected String getElmMin() {
		return ELM_MIN_IP;
	}

	/**
	 * Gets the maximum element local name.
	 *
	 * @return Maximum element local name.
	 */
	@Override
	protected String getElmMax() {
		return ELM_MAX_IP;
	}

	/**
	 * Gets the {@link Logger} to use.
	 *
	 * @return {@link Logger} instance to use for logging.
	 */
	@Override
	protected Logger getLogger() {
		return cat;
	}

}
