/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryExceedMaxExDdate} defines the action taken by the server
 * when executing commands that will result in an expiration date that exceeds
 * the maximum expiration date. The required "command" attribute is used to
 * define the command with a renewal feature, such as "renew" or "transfer". New
 * commands can be defined that include a renewal feature, such as "sync".
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDomain
 */
public class EPPRegistryExceedMaxExDate implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryExceedMaxExDate.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Possible values for the {@code policy} attribute.
	 */
	public static enum Policy {

		/**
		 * The server will fail the renewal command when the expiration date
		 * exceeds the maximum expiration date. An example is if the maximum
		 * expiration date is 10 years, and a client renews a domain name to 10.5
		 * years, the server will fail the renew.
		 */
		fail,

		/**
		 * The server will clip the fractional period when the expiration date
		 * exceeds the maximum expiration date by a fraction of a period and will
		 * fail the renewal command when the expiration date exceeds the maximum
		 * expiration date by a whole period and above. An example is if the
		 * maximum expiration date is 10 years, and the client renews a domain to
		 * 10.5 years, the server will clip the .5 fractional year so that the
		 * domain name will expire exactly in 10 years.
		 */
		clip,

		/**
		 * The server will execute the command with the renewal feature disabled
		 * when the expiration date exceeds the maximum expiration date. This may
		 * be the case for a command like "transfer" that includes a renewal
		 * feature in [RFC5731].
		 */
		disableRenewal;
	}

	/**
	 * XML local name for {@code EPPRegistryExceedMaxExDate}.
	 */
	public static final String ELM_LOCALNAME = "exceedMaxExDate";

	/**
	 * XML root tag for {@code EPPRegistryExceedMaxExDate}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML attribute name for the {@code command} attribute.
	 */
	public static final String ATTR_COMMAND = "command";

	/**
	 * The exceed maximum expiration date policy.
	 */
	Policy policy = null;

	/**
	 * The command associated with the exceed maximum expiration date policy.
	 */
	String command = null;

	/**
	 * Default constructor. Must call {@link #setPolicy(Policy)} and
	 * {@link #setCommand(String)} before calling the
	 * {@link #encode(org.w3c.dom.Document)} method.
	 */
	public EPPRegistryExceedMaxExDate() {
	}

	/**
	 * Construct an instance of {@code EPPRegistryExceedMaxExDdate} with the
	 * required policy and command.
	 *
	 * @param aPolicy
	 *           Exceed maximum expiration date policy applied
	 * @param aCommand
	 *           Command that the policy is applied to
	 */
	public EPPRegistryExceedMaxExDate(Policy aPolicy, String aCommand) {
		this.policy = aPolicy;
		this.command = aCommand;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryDomain} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryDomain} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryExceedMaxExDate}
	 *               instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Validate required attributes
		if (this.policy == null) {
			throw new EPPEncodeException("Invalid state on EPPRegistryExceedMaxExDate.encode: policy is not set");

		}
		if (this.command == null) {
			throw new EPPEncodeException("Invalid state on EPPRegistryExceedMaxExDate.encode: command is not set");

		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Policy
		Text policyVal = aDocument.createTextNode(this.policy.toString());
		root.appendChild(policyVal);

		// Command
		root.setAttribute(ATTR_COMMAND, this.command);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryExceedMaxExDate} attributes from the aElement
	 * DOM Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryExceedMaxExDate}
	 *           from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// Policy
		String thePolicyStr = EPPUtil.decodeStringValue(aElement);
		this.policy = Policy.valueOf(thePolicyStr);

		// Command
		this.command = EPPUtil.decodeStringAttr(aElement, ATTR_COMMAND);
	}

	/**
	 * Clone {@code EPPRegistryExceedMaxExDate}.
	 * 
	 * @return clone of {@code EPPRegistryExceedMaxExDate}
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryExceedMaxExDate} compare.
	 * 
	 * @param aObject
	 *           {@code EPPRegistryExceedMaxExDate} instance to compare with
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryExceedMaxExDate)) {
			return false;
		}

		EPPRegistryExceedMaxExDate theComp = (EPPRegistryExceedMaxExDate) aObject;

		// Policy
		if (!EqualityUtil.equals(this.policy, theComp.policy)) {
			cat.error("EPPRegistryExceedMaxExDate.equals(): policy not equal");
			return false;
		}

		// Command
		if (!EqualityUtil.equals(this.command, theComp.command)) {
			cat.error("EPPRegistryExceedMaxExDate.equals(): command not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 * 
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the exceed maximum expiration date policy applied.
	 * 
	 * @return The maximum expiration date policy applied if defined;
	 *         {@code null} otherwise.
	 */
	public Policy getPolicy() {
		return this.policy;
	}

	/**
	 * Sets the exceed maximum expiration date policy applied.
	 * 
	 * @param aPolicy
	 *           The maximum expiration date policy to set
	 */
	public void setPolicy(Policy aPolicy) {
		this.policy = aPolicy;
	}

	/**
	 * Gets the command that the maximum expiration date policy is applied to.
	 * 
	 * @return The command name if defined; {@code null} otherwise.
	 */
	public String getCommand() {
		return this.command;
	}

	/**
	 * Sets the command that the maximum expiration date policy is applied to.
	 * 
	 * @param aCommand
	 *           That command name, such as "renew" or "transfer".
	 */
	public void setCommand(String aCommand) {
		this.command = aCommand;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
