/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Defines the DNS Security Extensions (DNSSEC) policies for the server. The
 * &lt;registry:dnssec&gt; element contains the following child elements: <br>
 * <br>
 * <ul>
 * <li>&lt;registry:dsDataInterface&gt; - defines the DS Data Interface, as
 * defined in RFC 5910, policies. Use {@link #getDs()} and
 * {@link #setDs(EPPRegistryDS)} to get and set this element.</li>
 * <li>&lt;registry:keyDataInterface&gt; - defines the Key Data Interface, as
 * defined in RFC 5910, policies. Use {@link #getKey()} and
 * {@link #setKey(EPPRegistryKey)} to get and set this element.</li>
 * <li>&lt;registry:maxSigLife&gt; - defines the maximum signature life
 * policies. Use {@link #getMaxSigLife()} and
 * {@link #setMaxSigLife(EPPRegistryMaxSig)} to get and set this element.</li>
 * <li>&lt;registry:urgent&gt; - whether the client can specify the urgent
 * attribute for DNSSEC updates with a default value of {@code false}.. Use
 * {@link #getUrgent()} and {@link #setUrgent(Boolean)} to get and set this
 * element.</li>
 * </ul>
 *
 * In one &lt;registry:dnssec&gt; element, only one of
 * &lt;registry:dsDataInterface&gt; or &lt;registry:keyDataInterface&gt; may
 * exist. <br>
 * <br>
 *
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDomain
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDS
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryKey
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryMaxSig
 */
public class EPPRegistryDNSSEC implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryDNSSEC.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Constant for the {@code EPPRegistryDNSSEC} local name
	 */
	public static final String ELM_LOCALNAME = "dnssec";

	/**
	 * Constant for the {@code EPPRegistryDNSSEC} (prefix and local name)
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/** XML tag name for the {@code urgent} attribute. */
	public static final String ELM_URGENT = "urgent";

	/**
	 * DS Data interface attributes
	 */
	private EPPRegistryDS ds = null;

	/**
	 * Key Data interface attributes
	 */
	private EPPRegistryKey key = null;

	/**
	 * Max signature life attributes
	 */
	private EPPRegistryMaxSig maxSigLife = null;

	/**
	 * Whether client can specify the urgent attribute for DNSSEC update. Default
	 * value is {@code false}.
	 */
	private Boolean urgent = Boolean.FALSE;

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryDNSSEC} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryDNSSEC} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryDNSSEC} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryDNSSEC.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// DS Data or Key Data
		if (this.ds != null) {
			EPPUtil.encodeComp(aDocument, root, this.ds);
		}
		else {
			EPPUtil.encodeComp(aDocument, root, this.key);
		}

		// Max Sig Life
		EPPUtil.encodeComp(aDocument, root, this.maxSigLife);

		// Urgent
		if (this.urgent == null) {
			this.urgent = Boolean.FALSE;
		}
		EPPUtil.encodeString(aDocument, root, this.urgent.toString(),
		      EPPRegistryMapFactory.NS, EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_URGENT);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryDNSSEC} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryDNSSEC} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// DS Data
		this.ds = (EPPRegistryDS) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, EPPRegistryDS.ELM_NAME,
		      EPPRegistryDS.class);

		// Key Data
		this.key = (EPPRegistryKey) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, EPPRegistryKey.ELM_NAME,
		      EPPRegistryKey.class);

		// Max Sig Life
		this.maxSigLife = (EPPRegistryMaxSig) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryMaxSig.ELM_NAME, EPPRegistryMaxSig.class);

		// Urgent
		this.urgent = EPPUtil.decodeBoolean(aElement, EPPRegistryMapFactory.NS, ELM_URGENT);
		if (this.urgent == null) {
			this.urgent = Boolean.FALSE;
		}
	}

	/**
	 * Validate the state of the {@code EPPRegistryDNSSEC} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 */
	void validateState() throws EPPCodecException {
		if ((this.ds == null && this.key == null) || (this.ds != null && this.key != null)) {
			throw new EPPCodecException("only one of dsDataInterface or keyDataInterface element is required");
		}

		if (this.maxSigLife == null) {
			throw new EPPCodecException("maxSigLife element is not set");
		}
	}

	/**
	 * Clone {@code EPPRegistryDNSSEC}.
	 *
	 * @return clone of {@code EPPRegistryDNSSEC}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryDNSSEC clone = (EPPRegistryDNSSEC) super.clone();

		// DS Data
		if (this.ds != null) {
			clone.ds = (EPPRegistryDS) this.ds.clone();
		}

		// Key Data
		if (this.key != null) {
			clone.key = (EPPRegistryKey) this.key.clone();
		}

		// Max Sig Life
		if (this.maxSigLife != null) {
			clone.maxSigLife = (EPPRegistryMaxSig) this.maxSigLife.clone();
		}

		return clone;
	}

	/**
	 * implements a deep {@code EPPRegistryDNSSEC} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryDNSSEC} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryDNSSEC)) {
			return false;
		}

		EPPRegistryDNSSEC theComp = (EPPRegistryDNSSEC) aObject;

		// DS Data
		if (!EqualityUtil.equals(this.ds, theComp.ds)) {
			cat.error("EPPRegistryDNSSEC.equals(): ds not equal");
			return false;
		}

		// Key Data
		if (!EqualityUtil.equals(this.key, theComp.key)) {
			cat.error("EPPRegistryDNSSEC.equals(): key not equal");
			return false;
		}

		// Max Sig Life
		if (!EqualityUtil.equals(this.maxSigLife, theComp.maxSigLife)) {
			cat.error("EPPRegistryDNSSEC.equals(): maxSigLife not equal");
			return false;
		}

		// Urgent
		if (!EqualityUtil.equals(this.urgent, theComp.urgent)) {
			cat.error("EPPRegistryDNSSEC.equals(): urgent not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets DS Data interface attributes.
	 *
	 * @return instance of {@link EPPRegistryDS}
	 */
	public EPPRegistryDS getDs() {
		return this.ds;
	}

	/**
	 * Sets DS Data interface attributes.
	 *
	 * @param aDs
	 *           instance of {@link EPPRegistryDS}
	 */
	public void setDs(EPPRegistryDS aDs) {
		this.ds = aDs;
	}

	/**
	 * Gets Key Data interface attributes.
	 *
	 * @return instance of {@link EPPRegistryKey}
	 */
	public EPPRegistryKey getKey() {
		return this.key;
	}

	/**
	 * Sets Key Data interface attributes.
	 *
	 * @param aKey
	 *           instance of {@link EPPRegistryKey}
	 */
	public void setKey(EPPRegistryKey aKey) {
		this.key = aKey;
	}

	/**
	 * Gets max signature life policy.
	 *
	 * @return instance of {@link EPPRegistryMaxSig}
	 */
	public EPPRegistryMaxSig getMaxSigLife() {
		return this.maxSigLife;
	}

	/**
	 * Sets max signature life policy.
	 *
	 * @param aMaxSigLife
	 *           instance of {@link EPPRegistryMaxSig}
	 */
	public void setMaxSigLife(EPPRegistryMaxSig aMaxSigLife) {
		this.maxSigLife = aMaxSigLife;
	}

	/**
	 * Gets the urgent flag.
	 *
	 * @return {@code true} if the client can specify the urgent attribute for
	 *         DNSSEC updates. {@code false} if the client CANNOT specify the
	 *         urgent attribute for DNSSEC updates.
	 */
	public Boolean getUrgent() {
		return this.urgent;
	}

	/**
	 * Sets the urgent flag.
	 *
	 * @param aUrgent
	 *           {@code true} if the client can specify the urgent attribute for
	 *           DNSSEC updates. {@code false} if the client CANNOT specify the
	 *           urgent attribute for DNSSEC updates.
	 */
	public void setUrgent(Boolean aUrgent) {
		this.urgent = aUrgent;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
