/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import java.util.Date;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents the summary info of a zone object. Zone summary info of all
 * supported zones is returned is a list when the server receives an
 * &lt;info&gt; command, with a &lt;registry:all&gt; element in it.<br>
 * <br>
 *
 *
 * The zone summary info element contains:
 *
 * <ul>
 * <li>&lt;registry:name&gt; - fully qualified name of the zone. Use
 * {@link #setName(String)} and {@link #getName()} to access the attribute.</li>
 * <li>&lt;registry:crDate&gt; - date of zone object creation. Use
 * {@link #setCreateDate(Date)} and {@link #getCreateDate()} to access the
 * attribute.</li>
 * <li>&lt;registry:upDate&gt; - optional date of last update. Use
 * {@link #setUpdateDate(Date)} and {@link #getUpdateDate()} to access the
 * attribute.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryZoneList
 */
public class EPPRegistryZoneSummary implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryZoneSummary.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryZoneSummary}.
	 */
	public static final String ELM_LOCALNAME = "zone";

	/**
	 * XML root tag for {@code EPPRegistryZoneSummary}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code createDate} attribute.
	 */
	final static String ELM_CREATE_DATE = "crDate";

	/**
	 * XML Element Name of {@code updateDate} attribute.
	 */
	final static String ELM_UPDATE_DATE = "upDate";

	/**
	 * Fully qualified name of the zone object
	 */
	private EPPRegistryZoneName name;

	/**
	 * Date of object creation
	 */
	private Date createDate;

	/**
	 * Date of last update
	 */
	private Date updateDate;

	/**
	 * Default constructor. Attributes are set to:
	 * <ul>
	 * <li>{@code name} - null</li>
	 * <li>{@code createDate} - null</li>
	 * <li>{@code updateDate} - null</li>
	 * </ul>
	 *
	 * Use {@link #setName(EPPRegistryZoneName)} and {@link #setCreateDate(Date)}
	 * before calling {@link #encode(Document)}
	 */
	public EPPRegistryZoneSummary() {
		super();
	}

	/**
	 * Construct {@code EPPRegistryZoneSummary} with aLabel name and create date.
	 * {@code updateDate} is set to null.
	 *
	 * @param aName
	 *           fully qualified aLabel name of zone object
	 * @param aCreateDate
	 *           creation date of zone object
	 */
	public EPPRegistryZoneSummary(String aName, Date aCreateDate) {
		this();
		this.setName(aName);
		this.createDate = aCreateDate;
	}

	/**
	 * Construct {@code EPPRegistryZoneSummary} with name and create date.
	 * {@code updateDate} is set to null.
	 *
	 * @param aName
	 *           fully qualified name of zone object
	 * @param aCreateDate
	 *           creation date of zone object
	 */
	public EPPRegistryZoneSummary(EPPRegistryZoneName aName, Date aCreateDate) {
		this();
		this.setName(aName);
		this.createDate = aCreateDate;
	}

	/**
	 * Construct {@code EPPRegistryZoneSummary} with aLabel name, create date and last
	 * update date.
	 *
	 * @param aName
	 *           fully qualified aLabel name of zone object
	 * @param aCreateDate
	 *           creation date of zone object
	 * @param aUpdateDate
	 *           date of last update
	 */
	public EPPRegistryZoneSummary(String aName, Date aCreateDate, Date aUpdateDate) {
		this(aName, aCreateDate);
		this.updateDate = aUpdateDate;
	}

	/**
	 * Construct {@code EPPRegistryZoneSummary} with name, create date and last update
	 * date.
	 *
	 * @param aName
	 *           fully qualified name of zone object
	 * @param aCreateDate
	 *           creation date of zone object
	 * @param aUpdateDate
	 *           date of last update
	 */
	public EPPRegistryZoneSummary(EPPRegistryZoneName aName, Date aCreateDate, Date aUpdateDate) {
		this(aName, aCreateDate);
		this.updateDate = aUpdateDate;
	}

	/**
	 * Validate the state of the {@code EPPRegistryZoneSummary} instance. A valid state
	 * means that all of the required attributes have been set. If validateState
	 * returns without an exception, the state is valid. If the state is not
	 * valid, the EPPCodecException will contain a description of the error.
	 * throws EPPCodecException State error. This will contain the name of the
	 * attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 */
	void validateState() throws EPPCodecException {
		if (this.name == null) {
			throw new EPPCodecException("name required attribute is not set");
		}
		if (this.createDate == null) {
			throw new EPPCodecException("createDate required attribute is not set");
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryZoneSummary} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the {@code EPPRegistryZoneSummary}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryZoneSummary} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			cat.error(this.getClass().getName() + ".encode(): Invalid state on encode: " + e);
			throw new EPPEncodeException("Invalid state on " + this.getClass().getName() + ".encode(): " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Zone Name
		EPPUtil.encodeComp(aDocument, root, this.name);

		// Create Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.createDate, EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_CREATE_DATE);

		// Update Date
		if (this.updateDate != null) {
			EPPUtil.encodeTimeInstant(aDocument, root, this.updateDate, EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_UPDATE_DATE);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryZoneSummary} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryZoneSummary} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Zone Name
		this.name = (EPPRegistryZoneName) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryZoneName.ELM_ZONE_NAME, EPPRegistryZoneName.class);

		// Create Date
		this.createDate = EPPUtil.decodeTimeInstant(aElement, EPPRegistryMapFactory.NS, ELM_CREATE_DATE);

		// Update Date
		this.updateDate = EPPUtil.decodeTimeInstant(aElement, EPPRegistryMapFactory.NS, ELM_UPDATE_DATE);
	}

	/**
	 * implements a deep {@code EPPRegistryZoneSummary} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryZoneSummary} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryZoneSummary)) {
			return false;
		}

		EPPRegistryZoneSummary theComp = (EPPRegistryZoneSummary) aObject;

		// Zone Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryZoneSummary.equals(): name not equal");
			return false;
		}

		// Create Date
		if (!EqualityUtil.equals(this.createDate, theComp.createDate)) {
			cat.error("EPPRegistryZoneSummary.equals(): createDate not equal");
			return false;
		}

		// Update Date
		if (!EqualityUtil.equals(this.updateDate, theComp.updateDate)) {
			cat.error("EPPRegistryZoneSummary.equals(): updateDate not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryZoneSummary}.
	 *
	 * @return clone of {@code EPPRegistryZoneSummary}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryZoneSummary clone = (EPPRegistryZoneSummary) super.clone();

		// Zone Name
		if (this.name != null) {
			clone.name = (EPPRegistryZoneName) this.name.clone();
		}
		
		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the zone name.
	 *
	 * @return Zone name if set; {@code null} otherwise.
	 */
	public EPPRegistryZoneName getName() {
		return this.name;
	}

	/**
	 * Sets the zone name
	 *
	 * @param aName
	 *           Zone name
	 */
	public void setName(EPPRegistryZoneName aName) {
		this.name = aName;
	}

	/**
	 * Sets the aLabel zone name
	 *
	 * @param aName
	 *           aLabel zone name
	 */
	public void setName(String aName) {
		this.name = new EPPRegistryZoneName(aName, EPPRegistryZoneName.Form.aLabel);
	}

	/**
	 * Get creation date of zone object.
	 *
	 * @return creation date of zone object
	 */
	public Date getCreateDate() {
		return this.createDate;
	}

	/**
	 * Set creation date of zone object.
	 *
	 * @param createDate
	 *           creation date of zone object
	 */
	public void setCreateDate(Date createDate) {
		this.createDate = createDate;
	}

	/**
	 * Get last update date of zone object.
	 *
	 * @return last update date of zone object
	 */
	public Date getUpdateDate() {
		return this.updateDate;
	}

	/**
	 * Set last update date of zone object.
	 *
	 * @param updateDate
	 *           last update date of zone object
	 */
	public void setUpdateDate(Date updateDate) {
		this.updateDate = updateDate;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
