/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import com.verisign.epp.codec.gen.EPPCodecException;

/**
 * Represents the period of time a domain object is in the pending transfer
 * before the transfer is auto approved by the server. The
 * &lt;registry:transferHoldPeriod&gt; element MUST have the "unit" attribute
 * with the possible values of "y" for year, "m" for month, and "d" for day..
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryDomain
 */
public class EPPRegistryTransferHoldPeriodType extends EPPRegistryPeriodType {

	/**
	 * XML local name for {@code EPPRegistryTransferHoldPeriodType}.
	 */
	public static final String ELM_LOCALNAME = "transferHoldPeriod";

	/**
	 * Default constructor. Must call {@link #setNumber(Integer)} and
	 * {@link #setUnit(EPPRegistryPeriodType.Unit)} before calling the
	 * {@link #encode(org.w3c.dom.Document)} method.
	 */
	public EPPRegistryTransferHoldPeriodType() {
	}

	/**
	 * Construct an instance of {@code EPPRegistryTransferHoldPeriodType} with
	 * {@code number} and {@code unit}.
	 *
	 * @param aNumber
	 *           number must be &gt; 0
	 * @param aUnit
	 *           unit must be one of {@code Unit.y}, {@code Unit.m}, or
	 *           {@code Unit.d}
	 */
	public EPPRegistryTransferHoldPeriodType(Integer aNumber, Unit aUnit) {
		super(aNumber, aUnit);
	}

	/**
	 * Construct an instance of {@code EPPRegistryTransferHoldPeriodType} with
	 * {@code number} and {@code unit}.
	 *
	 * @param aNumber
	 *           number must be &gt; 0
	 * @param aUnit
	 *           unit must be one of {@code Unit.y}, {@code Unit.m}, or
	 *           {@code Unit.d}
	 */
	public EPPRegistryTransferHoldPeriodType(int aNumber, Unit aUnit) {
		super(aNumber, aUnit);
	}

	/**
	 * Extra validation rules on top of {@link EPPRegistryPeriodType}:
	 * {@code number} must be greater than "0"; {@code unit} must be one of "y",
	 * "m" or "d".
	 */
	@Override
	void extraValidate() throws EPPCodecException {
		// Number
		int n = this.number.intValue();
		if (n <= 0) {
			throw new EPPCodecException(getRootName() + ": number should be greater than 0");
		}

		// Unit
		if (super.unit != Unit.y && super.unit != Unit.m && super.unit != Unit.d) {
			throw new EPPCodecException(getRootName() + ": invalid unit. Valid values: y/m/d");
		}
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Gets the root element local name.
	 *
	 * @return Root element local name.
	 */
	@Override
	protected String getRootName() {
		return ELM_LOCALNAME;
	}

}
