/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Defined the Registry Grace Period (RGP) status periods. The
 * &lt;registry:rgp&gt; element contains the following child elements, where
 * each child element supports the "unit" attribute with the possible values of
 * "y" for year, "m" for month, "d" for day, and "h" for hour: <br>
 * <br>
 * <ul>
 * <li>&lt;registry:redemptionPeriod&gt; - The length of time that a domain
 * object will remain in the redemptionPeriod status unless the restore request
 * command is received. Use {@link #getRedemptionPeriod()} and
 * {@link #setRedemptionPeriod(EPPRegistryRedemptionPeriodType)} to get and set
 * the element.</li>
 * <li>&lt;registry:pendingRestore&gt; - The length of time that the domain
 * object will remain in the pendingRestore status unless the restore report
 * command is received. Use {@link #getPendingRestorePeriod()} and
 * {@link #setPendingRestorePeriod(EPPRegistryPendingRestorePeriodType)} to get
 * and set the element.</li>
 * <li>&lt;registry:pendingDelete&gt; - The length of time that the domain
 * object will remain in the pendingDelete status prior to be purged. Use
 * {@link #getPendingDeletePeriod()} and
 * {@link #setPendingDeletePeriod(EPPRegistryPendingDeletePeriodType)} to get
 * and set the element.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryDomain
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryRedemptionPeriodType
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryPendingRestorePeriodType
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryPendingDeletePeriodType
 */
public class EPPRegistryRGP implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryRGP.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryRGP}.
	 */
	public static final String ELM_LOCALNAME = "rgp";

	/**
	 * XML root tag for {@code EPPRegistryRGP}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * The length of time that a domain object will remain in the
	 * redemptionPeriod status unless the restore request command is received.
	 */
	private EPPRegistryRedemptionPeriodType redemptionPeriod = null;

	/**
	 * The length of time that the domain object will remain in the
	 * pendingRestore status unless the restore report command is received.
	 */
	private EPPRegistryPendingRestorePeriodType pendingRestorePeriod = null;

	/**
	 * The length of time that the domain object will remain in the pendingDelete
	 * status prior to being purged.
	 */
	private EPPRegistryPendingDeletePeriodType pendingDeletePeriod = null;

	/**
	 * Default constructor. All attributes are initialized to {@code null}. Must
	 * call {@link #setRedemptionPeriod(EPPRegistryRedemptionPeriodType)},
	 * {@link #setPendingRestorePeriod(EPPRegistryPendingRestorePeriodType)} and
	 * {@link #setPendingDeletePeriod(EPPRegistryPendingDeletePeriodType)} before
	 * calling {@link #encode(Document)} method.
	 */
	public EPPRegistryRGP() {
	}

	/**
	 * Constructor for {@code EPPRegistryRGP} that takes all of the attribute
	 * values as parameters.
	 *
	 * @param aRedemptionPeriod
	 *           The length of time that a domain object will remain in the
	 *           redemptionPeriod status unless the restore request command is
	 *           received.
	 * @param aPendingRestorePeriod
	 *           The length of time that the domain object will remain in the
	 *           pendingRestore status unless the restore report command is
	 *           received.
	 * @param aPendingDeletePeriod
	 *           The length of time that the domain object will remain in the
	 *           pendingDelete status prior to being purged.
	 */
	public EPPRegistryRGP(EPPRegistryRedemptionPeriodType aRedemptionPeriod,
	      EPPRegistryPendingRestorePeriodType aPendingRestorePeriod,
	      EPPRegistryPendingDeletePeriodType aPendingDeletePeriod) {
		this.redemptionPeriod = aRedemptionPeriod;
		this.pendingRestorePeriod = aPendingRestorePeriod;
		this.pendingDeletePeriod = aPendingDeletePeriod;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryRGP} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the {@code EPPRegistryRGP}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryRGP} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryRGP.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Redemption Period
		EPPUtil.encodeComp(aDocument, root, this.redemptionPeriod);

		// Pending Restore Period
		EPPUtil.encodeComp(aDocument, root, this.pendingRestorePeriod);

		// Pending Delete Period
		EPPUtil.encodeComp(aDocument, root, this.pendingDeletePeriod);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryRGP} attributes from the aElement DOM Element
	 * tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryRGP} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Redemption Period
		this.redemptionPeriod = (EPPRegistryRedemptionPeriodType) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryRedemptionPeriodType.ELM_LOCALNAME, EPPRegistryRedemptionPeriodType.class);

		// Pending Restore Period
		this.pendingRestorePeriod = (EPPRegistryPendingRestorePeriodType) EPPUtil.decodeComp(aElement,
		      EPPRegistryMapFactory.NS, EPPRegistryPendingRestorePeriodType.ELM_LOCALNAME,
		      EPPRegistryPendingRestorePeriodType.class);

		// Pending Delete Period
		this.pendingDeletePeriod = (EPPRegistryPendingDeletePeriodType) EPPUtil.decodeComp(aElement,
		      EPPRegistryMapFactory.NS, EPPRegistryPendingDeletePeriodType.ELM_LOCALNAME,
		      EPPRegistryPendingDeletePeriodType.class);
	}

	/**
	 * Validate the state of the {@code EPPRegistryRGP} instance. A valid state
	 * means that all of the required attributes have been set. If validateState
	 * returns without an exception, the state is valid. If the state is not
	 * valid, the EPPCodecException will contain a description of the error.
	 * throws EPPCodecException State error. This will contain the name of the
	 * attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	void validateState() throws EPPCodecException {
		if (this.redemptionPeriod == null) {
			throw new EPPCodecException("redemptionPeriod element is not set");
		}
		if (this.pendingRestorePeriod == null) {
			throw new EPPCodecException("pendingRestorePeriod element is not set");
		}
		if (this.pendingDeletePeriod == null) {
			throw new EPPCodecException("pendingDeletePeriod element is not set");
		}
	}

	/**
	 * Clone {@code EPPRegistryRGP}.
	 *
	 * @return clone of {@code EPPRegistryRGP}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryRGP clone = (EPPRegistryRGP) super.clone();

		// Redemption Period
		if (this.redemptionPeriod != null) {
			clone.redemptionPeriod = (EPPRegistryRedemptionPeriodType) this.redemptionPeriod.clone();
		}

		// Pending Restore Period
		if (this.pendingRestorePeriod != null) {
			clone.pendingRestorePeriod = (EPPRegistryPendingRestorePeriodType) this.pendingRestorePeriod.clone();
		}

		// Pending Delete Period
		if (this.pendingDeletePeriod != null) {
			clone.pendingDeletePeriod = (EPPRegistryPendingDeletePeriodType) this.pendingDeletePeriod.clone();
		}

		return clone;
	}

	/**
	 * implements a deep {@code EPPRegistryRGP} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryRGP} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryRGP)) {
			return false;
		}

		EPPRegistryRGP theComp = (EPPRegistryRGP) aObject;

		// Redmption Period
		if (!EqualityUtil.equals(this.redemptionPeriod, theComp.redemptionPeriod)) {
			cat.error("EPPRegistryRGP.equals(): redemptionPeriod not equal");
			return false;
		}

		// Pending Restore Period
		if (!EqualityUtil.equals(this.pendingRestorePeriod, theComp.pendingRestorePeriod)) {
			cat.error("EPPRegistryRGP.equals(): pendingRestorePeriod not equal");
			return false;
		}

		// Pending Delete Period
		if (!EqualityUtil.equals(this.pendingDeletePeriod, theComp.pendingDeletePeriod)) {
			cat.error("EPPRegistryRGP.equals(): pendingDeletePeriod not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the redemption period.
	 *
	 * @return instance of {@link EPPRegistryRedemptionPeriodType} that defines
	 *         redemption period attributes.
	 */
	public EPPRegistryRedemptionPeriodType getRedemptionPeriod() {
		return this.redemptionPeriod;
	}

	/**
	 * Sets the redemption period.
	 *
	 * @param aRedemptionPeriod
	 *           instance of {@link EPPRegistryRedemptionPeriodType} that defines
	 *           redemption period attributes.
	 */
	public void setRedemptionPeriod(EPPRegistryRedemptionPeriodType aRedemptionPeriod) {
		this.redemptionPeriod = aRedemptionPeriod;
	}

	/**
	 * Gets the pending delete period.
	 *
	 * @return instance of {@link EPPRegistryPendingDeletePeriodType} that
	 *         defines pending delete period attributes.
	 */
	public EPPRegistryPendingDeletePeriodType getPendingDeletePeriod() {
		return this.pendingDeletePeriod;
	}

	/**
	 * Sets the pending delete period.
	 *
	 * @param aPendingDeletePeriod
	 *           instance of {@link EPPRegistryPendingDeletePeriodType} that
	 *           defines pending delete period attributes.
	 */
	public void setPendingDeletePeriod(EPPRegistryPendingDeletePeriodType aPendingDeletePeriod) {
		this.pendingDeletePeriod = aPendingDeletePeriod;
	}

	/**
	 * Gets the pending restore period.
	 *
	 * @return instance of {@link EPPRegistryPendingRestorePeriodType} that
	 *         defines pending restore period attributes.
	 */
	public EPPRegistryPendingRestorePeriodType getPendingRestorePeriod() {
		return this.pendingRestorePeriod;
	}

	/**
	 * Sets the pending restore period.
	 *
	 * @param aPendingRestorePeriod
	 *           instance of {@link EPPRegistryPendingRestorePeriodType} that
	 *           defines pending restore period attributes.
	 */
	public void setPendingRestorePeriod(EPPRegistryPendingRestorePeriodType aPendingRestorePeriod) {
		this.pendingRestorePeriod = aPendingRestorePeriod;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
