/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.util.Date;

import com.verisign.epp.codec.gen.EPPCodec;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents an EPP Registry &lt;registry:creData&gt; response to a
 * {@code EPPRegistryCreateCmd}. When a &lt;crate&gt; command has been processed
 * successfully, the EPP &lt;resData&gt; element MUST contain a child
 * &lt;registry:creData&gt; element that identifies the registry namespace and
 * the location of the registry schema. The &lt;registry:creData&gt; element
 * contains the following child elements:<br>
 * <br>
 * 
 * <ul>
 * <li>A &lt;registry:name&gt; element that contains the fully qualified name of
 * zone object that has been created. Use {@code getName} and {@code setName} to
 * get and set the element.</li>
 * <li>A &lt;registry.crDate&gt; element that contains the date and time of zone
 * object creation. Use {@code getCreateDate} and {@code setCreateDate} to get
 * and set the element.</li>
 * </ul>
 * 
 * <br>
 * <br>
 * 
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryCreateCmd
 */
public class EPPRegistryCreateResp extends EPPResponse {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryCheckResult.class);

	/**
	 * XML local name for {@code EPPRegistryCreateResp}.
	 */
	public static final String ELM_LOCALNAME = "creData";

	/**
	 * XML root tag for {@code EPPRegistryCreateResp}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name for the {@code crDate} attribute.
	 */
	private final static String ELM_CREATION_DATE = "crDate";

	/**
	 * Name of the zone object created.
	 */
	private EPPRegistryZoneName name = null;

	/**
	 * {@code Date} of zone object creation.
	 */
	private Date createDate = null;

	/**
	 * {@code EPPRegistryCreateResp} default constructor. Must call required
	 * setter methods before encode. the defaults include the following:<br>
	 * <br>
	 * 
	 * <ul>
	 * <li>{@code name} is set to {@code null}</li>
	 * <li>{@code createDate} date is set to {@code null}</li>
	 * </ul>
	 * 
	 * <br>
	 * {@code name} must be set before invoking {@code encode}.
	 */
	public EPPRegistryCreateResp() {
	}

	/**
	 * {@code EPPRegistryCreateResp} constructor that takes the aLabel
	 * {@code name} value as a parameter. The creation date must be set before
	 * invoking {@code #encode(Document)}.
	 * 
	 * @param aTransId
	 *           transaction Id associated with response
	 * @param aName
	 *           aLabel name of zone object
	 */
	public EPPRegistryCreateResp(EPPTransId aTransId, String aName) {
		super(aTransId);
		this.setName(aName);
	}

	/**
	 * {@code EPPRegistryCreateResp} constructor that takes the {@code name}
	 * value as a parameter. The creation date must be set before invoking
	 * {@code #encode(Document)}.
	 * 
	 * @param aTransId
	 *           transaction Id associated with response
	 * @param aName
	 *           name of zone object
	 */
	public EPPRegistryCreateResp(EPPTransId aTransId, EPPRegistryZoneName aName) {
		super(aTransId);
		this.setName(aName);
	}

	/**
	 * {@code EPPRegistryCreateResp} constructor that takes required values as
	 * parameters.
	 * 
	 * @param aTransId
	 *           transaction Id associated with response
	 * @param aName
	 *           aLabel name of zone object
	 * @param aCreateDate
	 *           creation date of the zone object
	 * 
	 */
	public EPPRegistryCreateResp(EPPTransId aTransId, String aName, Date aCreateDate) {
		this(aTransId, aName);
		this.createDate = aCreateDate;
	}

	/**
	 * {@code EPPRegistryCreateResp} constructor that takes required values as
	 * parameters.
	 * 
	 * @param aTransId
	 *           transaction Id associated with response
	 * @param aName
	 *           name of zone object
	 * @param aCreateDate
	 *           creation date of the zone object
	 * 
	 */
	public EPPRegistryCreateResp(EPPTransId aTransId, EPPRegistryZoneName aName, Date aCreateDate) {
		this(aTransId, aName);
		this.createDate = aCreateDate;
	}

	/**
	 * Gets the EPP command type associated with {@code EPPRegistryCreateResp}.
	 * 
	 * @return {@code EPPRegistryCreateResp.ELM_NAME}
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * {@code EPPRegistryCreateResp}.
	 * 
	 * @return {@code EPPRegistryMapFactory.NS}
	 */
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Compare an instance of {@code EPPRegistryCreateResp} with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryCreateResp)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPRegistryCreateResp theComp = (EPPRegistryCreateResp) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryCreateResp.equals(): name not equal");
			return false;
		}

		// Create Date
		if (!EqualityUtil.equals(this.createDate, theComp.createDate)) {
			cat.error("EPPRegistryCreateResp.equals(): createDate not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryCreateResp}.
	 * 
	 * @return clone of {@code EPPRegistryCreateResp}
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryCreateResp clone = (EPPRegistryCreateResp) super.clone();

		// Name
		if (this.name != null) {
			clone.name = (EPPRegistryZoneName) this.name.clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 * 
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Validate the state of the {@code EPPRegistryCreateResp} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 * 
	 * @throws EPPCodecException
	 */
	void validateState() throws EPPCodecException {
		if (name == null) {
			throw new EPPCodecException("name required attribute is not set");
		}

		if (createDate == null) {
			throw new EPPCodecException("required attribute createDate is not set");
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryCreateResp} instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryCreateResp} instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryCreateResp} instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		// Validate Attributes
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryCreateResp.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		root.setAttribute("xmlns:registry", EPPRegistryMapFactory.NS);

		// Name
		EPPUtil.encodeComp(aDocument, root, this.name);

		// Creation Date
		EPPUtil.encodeTimeInstant(aDocument, root, createDate, EPPRegistryMapFactory.NS, ELM_CREATION_DATE);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryCreateResp} attributes from the
	 * {@code aElement} DOM Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryCreateResp} from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode {@code aElement}
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = (EPPRegistryZoneName) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryZoneName.ELM_ZONE_NAME, EPPRegistryZoneName.class);

		// Creation Date
		createDate = EPPUtil.decodeTimeInstant(aElement, EPPRegistryMapFactory.NS, ELM_CREATION_DATE);

		// Validate Attributes
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPDecodeException("Invalid state on EPPRegistryCreateResp.decode: " + e);
		}
	}

	/**
	 * Gets the zone name.
	 *
	 * @return Zone name if set; {@code null} otherwise.
	 */
	public EPPRegistryZoneName getName() {
		return this.name;
	}

	/**
	 * Sets the zone name
	 *
	 * @param aName
	 *           Zone name
	 */
	public void setName(EPPRegistryZoneName aName) {
		this.name = aName;
	}

	/**
	 * Sets the aLabel zone name
	 *
	 * @param aName
	 *           aLabel zone name
	 */
	public void setName(String aName) {
		this.name = new EPPRegistryZoneName(aName, EPPRegistryZoneName.Form.aLabel);
	}

	/**
	 * Get the zone object creation date.
	 * 
	 * @return zone object creation date
	 */
	public Date getCreateDate() {
		return createDate;
	}

	/**
	 * Set the zone object creation date.
	 * 
	 * @param createDate
	 *           zone object creation date
	 */
	public void setCreateDate(Date createDate) {
		this.createDate = createDate;
	}
}
