/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.orgext;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPOrgExtCreate} is used in the extension to the update command to
 * add, remove, and change organizations.
 */
public class EPPOrgExtUpdate implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPOrgExtUpdate.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPOrgExtCreate}.
	 */
	public static final String ELM_LOCALNAME = "update";

	/**
	 * XML root tag for {@code EPPOrgExtCreate}.
	 */
	public static final String ELM_NAME = EPPOrgExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Organizations to add
	 */
	EPPOrgExtAddRemChg addOrgs = null;

	/**
	 * Organizations to remove
	 */
	EPPOrgExtAddRemChg remOrgs = null;

	/**
	 * Organizations to change
	 */
	EPPOrgExtAddRemChg chgOrgs = null;

	/**
	 * Default constructor for {@code EPPOrgExtCreate}.
	 */
	public EPPOrgExtUpdate() {
	}

	/**
	 * constructor for {@code EPPOrgExtCreate} that takes the three set of
	 * organizations to add, remove, and change as parameters. A parameter can be
	 * set to {@code null} to be undefined.
	 * 
	 * @param aAddOrgs
	 *           Organizations to add. Set to {@code null} to be undefined.
	 * @param aRemOrgs
	 *           Organizations to remove. Set to {@code null} to be undefined.
	 * @param aChgOrgs
	 *           Organizations to change. Set to {@code null} to be undefined.
	 * 
	 */
	public EPPOrgExtUpdate(EPPOrgExtAddRemChg aAddOrgs, EPPOrgExtAddRemChg aRemOrgs, EPPOrgExtAddRemChg aChgOrgs) {
		this.addOrgs = aAddOrgs;
		this.remOrgs = aRemOrgs;
		this.chgOrgs = aChgOrgs;
	}

	/**
	 * Gets the organizations to add.
	 * 
	 * @return The organizations to add if defined; {@code null} otherwise.
	 */
	public EPPOrgExtAddRemChg getAddOrgs() {
		return this.addOrgs;
	}

	/**
	 * Sets the organizations to add.
	 * 
	 * @param aAddOrgs
	 *           The organizations to add. Set to {@code null} to clear the
	 *           organizations.
	 */
	public void setAddOrgs(EPPOrgExtAddRemChg aAddOrgs) {
		this.addOrgs = aAddOrgs;
	}

	/**
	 * Adds an organization to the list of organizations to add.
	 * 
	 * @param aOrgId
	 *           Organization to add
	 */
	public void addAddOrg(EPPOrgExtId aOrgId) {
		if (this.addOrgs == null) {
			this.addOrgs = new EPPOrgExtAddRemChg(EPPOrgExtAddRemChg.Action.ADD);
		}

		this.addOrgs.addOrgId(aOrgId);
	}

	/**
	 * Are any add organizations defined?
	 * 
	 * @return {@code true} if add organizations are defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasAddOrgs() {
		return this.addOrgs != null && this.addOrgs.hasOrgIds();
	}

	/**
	 * Gets the organizations to remove.
	 * 
	 * @return The organizations to remove if defined; {@code null} otherwise.
	 */
	public EPPOrgExtAddRemChg getRemOrgs() {
		return this.remOrgs;
	}

	/**
	 * Sets the organizations to remove.
	 * 
	 * @param aRemOrgs
	 *           The organizations to remove. Set to {@code null} to clear the
	 *           organizations.
	 */
	public void setRemOrgs(EPPOrgExtAddRemChg aRemOrgs) {
		this.remOrgs = aRemOrgs;
	}

	/**
	 * Adds an organization to the list of organizations to remove.
	 * 
	 * @param aOrgId
	 *           Organization to remove
	 */
	public void addRemOrg(EPPOrgExtId aOrgId) {
		if (this.remOrgs == null) {
			this.remOrgs = new EPPOrgExtAddRemChg(EPPOrgExtAddRemChg.Action.REM);
		}

		this.remOrgs.addOrgId(aOrgId);
	}

	/**
	 * Are any remove organizations defined?
	 * 
	 * @return {@code true} if remove organizations are defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasRemOrgs() {
		return this.remOrgs != null && this.remOrgs.hasOrgIds();
	}

	/**
	 * Gets the organizations to change.
	 * 
	 * @return The organizations to change if defined; {@code null} otherwise.
	 */
	public EPPOrgExtAddRemChg getChgOrgs() {
		return this.chgOrgs;
	}

	/**
	 * Sets the organizations to change.
	 * 
	 * @param aChgOrgs
	 *           The organizations to change. Set to {@code null} to clear the
	 *           organizations.
	 */
	public void setChgOrgs(EPPOrgExtAddRemChg aChgOrgs) {
		this.chgOrgs = aChgOrgs;
	}

	/**
	 * Adds an organization to the list of organizations to change.
	 * 
	 * @param aOrgId
	 *           Organization to change
	 */
	public void addChgOrg(EPPOrgExtId aOrgId) {
		if (this.chgOrgs == null) {
			this.chgOrgs = new EPPOrgExtAddRemChg(EPPOrgExtAddRemChg.Action.CHG);
		}

		this.chgOrgs.addOrgId(aOrgId);
	}

	/**
	 * Are any change organizations defined?
	 * 
	 * @return {@code true} if change organizations are defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasChgOrgs() {
		return this.chgOrgs != null && this.chgOrgs.hasOrgIds();
	}

	/**
	 * encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is updated and each instance attribute is
	 * appended as a child node.
	 * 
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 * 
	 * @return Element Root element associated with the object
	 * 
	 * @exception EPPEncodeException
	 *               Error encoding {@code EPPOrgExtCreate}
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (!this.hasAddOrgs() && !this.hasRemOrgs() && !this.hasChgOrgs()) {
			throw new EPPEncodeException("Empty add, remove, and change organizations in EPPOrgExtUpdate");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgExtFactory.NS, ELM_NAME);

		// Add organizations
		if (this.hasAddOrgs()) {
			EPPUtil.encodeComp(aDocument, root, this.addOrgs);
		}

		// Remove organizations
		if (this.hasRemOrgs()) {
			EPPUtil.encodeComp(aDocument, root, this.remOrgs);
		}

		// Change organizations
		if (this.hasChgOrgs()) {
			EPPUtil.encodeComp(aDocument, root, this.chgOrgs);
		}

		return root;
	}

	/**
	 * decode a DOM element tree to initialize the instance attributes. The
	 * {@code aElement} argument represents the root DOM element and is used to
	 * traverse the DOM nodes for instance attribute values.
	 * 
	 * @param aElement
	 *           {@code Element} to decode
	 * 
	 * @exception EPPDecodeException
	 *               Error decoding {@code Element}
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// Add organizations
		this.addOrgs = (EPPOrgExtAddRemChg) EPPUtil.decodeComp(aElement, EPPOrgExtFactory.NS,
		      EPPOrgExtAddRemChg.ELM_ADD_NAME, EPPOrgExtAddRemChg.class);

		// Remove organizations
		this.remOrgs = (EPPOrgExtAddRemChg) EPPUtil.decodeComp(aElement, EPPOrgExtFactory.NS,
		      EPPOrgExtAddRemChg.ELM_REM_NAME, EPPOrgExtAddRemChg.class);

		// Change organizations
		this.chgOrgs = (EPPOrgExtAddRemChg) EPPUtil.decodeComp(aElement, EPPOrgExtFactory.NS,
		      EPPOrgExtAddRemChg.ELM_CHG_NAME, EPPOrgExtAddRemChg.class);

	}

	/**
	 * clone an {@code EPPCodecComponent}.
	 * 
	 * @return clone of concrete {@code EPPOrgExt}
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgExtUpdate clone = (EPPOrgExtUpdate) super.clone();

		// Add organizations
		if (this.addOrgs != null) {
			clone.addOrgs = (EPPOrgExtAddRemChg) this.addOrgs.clone();
		}

		// Remove organizations
		if (this.remOrgs != null) {
			clone.remOrgs = (EPPOrgExtAddRemChg) this.remOrgs.clone();
		}

		// Change organizations
		if (this.chgOrgs != null) {
			clone.chgOrgs = (EPPOrgExtAddRemChg) this.chgOrgs.clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 * 
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of {@code EPPOrgExt} with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgExtUpdate)) {
			return false;
		}

		EPPOrgExtUpdate other = (EPPOrgExtUpdate) aObject;

		// Add organizations
		if (!EqualityUtil.equals(this.addOrgs, other.addOrgs)) {
			cat.error("EPPOrgExtUpdate.equals(): addOrgs not equal");
			return false;
		}

		// Remove organizations
		if (!EqualityUtil.equals(this.remOrgs, other.remOrgs)) {
			cat.error("EPPOrgExtUpdate.equals(): remOrgs not equal");
			return false;
		}

		// Change organizations
		if (!EqualityUtil.equals(this.chgOrgs, other.chgOrgs)) {
			cat.error("EPPOrgExtUpdate.equals(): chgOrgs not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 * 
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPOrgExtFactory.NS;
	}

}
