/***********************************************************
Copyright (C) 2019 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.loginsec.v04;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Login Security User Agent identifies the client software and platform used by
 * the server to identify functional or security constraints, current security
 * issues, and potential future functional or security issues for the client.
 */
public class EPPLoginSecUserAgent implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPLoginSecUserAgent.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for <code>EPPLoginSecUserAgent</code>.
	 */
	public static final String ELM_LOCALNAME = "userAgent";

	/**
	 * XML root tag for <code>EPPLoginSecUserAgent</code>.
	 */
	public static final String ELM_NAME = EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML tag name for the <code>app</code> attribute.
	 */
	private final static String ELM_APP = "app";

	/**
	 * XML tag name for the <code>tech</code> attribute.
	 */
	private final static String ELM_TECH = "tech";

	/**
	 * XML tag name for the <code>os</code> attribute.
	 */
	private final static String ELM_OS = "os";

	/**
	 * OPTIONAL name of the client application software with version if
	 * available, such as the name of the client SDK "EPP SDK 1.0.0".
	 */
	private String app;

	/**
	 * OPTIONAL technology used for the client software with version if
	 * available, such as "Java 11.0.2".
	 */
	private String tech;

	/**
	 * OPTIONAL client operating system used, such as "x86_64 Mac OS X 10.21".
	 */
	private String os;

	/**
	 * Default constructor for <code>EPPLoginSecUserAgent</code>.
	 */
	public EPPLoginSecUserAgent() {
	}

	/**
	 * Constructor for <code>EPPLoginSecUserAgent</code> that sets all
	 * attributes.
	 * 
	 * @param aApp
	 *           OPTIONAL name of the client application software with version if
	 *           available, such as the name of the client SDK "EPP SDK 1.0.0".
	 *           An example is "EPP SDK 1.0.0". Set to {@code null} if undefined.
	 * @param aTech
	 *           OPTIONAL technology used for the client software with version if
	 *           available, such as "Java 11.0.2". Set to {@code null} if
	 *           undefined.
	 * @param aOs
	 *           OPTIONAL operating system used, such as "x86_64 Mac OS X 10.21".
	 *           Set to {@code null} if undefined.
	 */
	public EPPLoginSecUserAgent(String aApp, String aTech, String aOs) {
		this.app = aApp;
		this.tech = aTech;
		this.os = aOs;
	}

	/**
	 * Is the name of the client application software set?
	 *
	 * @return <code>true</code> if the name of the client application software
	 *         is defined; <code>false</code> otherwise.
	 */
	public boolean hasApp() {
		if (this.app != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the name of the client application software.
	 * 
	 * @return Gets the optional name of the client application software;
	 *         {@code null} otherwise.
	 */
	public String getApp() {
		return this.app;
	}

	/**
	 * Sets the name of the client application software.
	 * 
	 * @param aApp
	 *           Name of the client application software with version if
	 *           available, such as the name of the client SDK being used. An
	 *           example is "EPP SDK 1.0.0". Set to {@code null} if undefined.
	 */
	public void setApp(String aApp) {
		this.app = aApp;
	}

	/**
	 * Is the technology used for the client software set?
	 *
	 * @return <code>true</code> if the technology of the client software is
	 *         defined; <code>false</code> otherwise.
	 */
	public boolean hasTech() {
		if (this.tech != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the technology used for the client software.
	 * 
	 * @return Gets the optional technology used for the client software;
	 *         {@code null} otherwise.
	 */
	public String getTech() {
		return this.tech;
	}

	/**
	 * Sets the technology used for the client software.
	 * 
	 * @param aTech
	 *           Technology used for the client software, such as "Java 11.0.2".
	 *           Set to {@code null} if undefined.
	 */
	public void setTech(String aTech) {
		this.tech = aTech;
	}

	/**
	 * Is the client operating system set?
	 *
	 * @return <code>true</code> if the client operating system is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasOs() {
		if (this.os != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the client operating system being used.
	 * 
	 * @return Gets the operating system used, such as "x86_64 Mac OS X 10.21";
	 *         {@code null} otherwise.
	 */
	public String getOs() {
		return this.os;
	}

	/**
	 * Sets the client operating system being used.
	 * 
	 * @param aOs
	 *           Name of the client operating system, such as "x86_64 Mac OS X
	 *           10.21". Set to {@code null} if undefined.
	 */
	public void setOs(String aOs) {
		this.os = aOs;
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPLoginSecUserAgent</code>
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPLoginSecUserAgent.encode(Document)");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPLoginSecExtFactory.NS, ELM_NAME);

		// App
		EPPUtil.encodeString(aDocument, root, this.app, EPPLoginSecExtFactory.NS,
		      EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_APP);

		// Tech
		EPPUtil.encodeString(aDocument, root, this.tech, EPPLoginSecExtFactory.NS,
		      EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_TECH);

		// Os
		EPPUtil.encodeString(aDocument, root, this.os, EPPLoginSecExtFactory.NS,
		      EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_OS);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           <code>Element</code> to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// App
		this.app = EPPUtil.decodeString(aElement, EPPLoginSecExtFactory.NS, ELM_APP);

		// Tech
		this.tech = EPPUtil.decodeString(aElement, EPPLoginSecExtFactory.NS, ELM_TECH);

		// Os
		this.os = EPPUtil.decodeString(aElement, EPPLoginSecExtFactory.NS, ELM_OS);
	}

	/**
	 * Compare an instance of <code>EPPLoginSecUserAgent</code> with this
	 * instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLoginSecUserAgent)) {
			cat.error(
			      "EPPLoginSecUserAgent.equals(): " + aObject.getClass().getName() + " not EPPLoginSecUserAgent instance");

			return false;
		}

		EPPLoginSecUserAgent other = (EPPLoginSecUserAgent) aObject;

		// App
		if (!EqualityUtil.equals(this.app, other.app)) {
			cat.error("EPPLoginSecUserAgent.equals(): app not equal");
			return false;
		}

		// Tech
		if (!EqualityUtil.equals(this.tech, other.tech)) {
			cat.error("EPPLoginSecUserAgent.equals(): tech not equal");
			return false;
		}

		// Os
		if (!EqualityUtil.equals(this.os, other.os)) {
			cat.error("EPPLoginSecUserAgent.equals(): os not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone an <code>EPPCodecComponent</code> instance.
	 *
	 * @return clone of concrete <code>EPPLoginSecUserAgent</code>
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {

		EPPLoginSecUserAgent clone = (EPPLoginSecUserAgent) super.clone();

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPLoginSecExtFactory.NS;
	}

}
