/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Iterator;
import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.codec.host.EPPHostAddress;

/**
 * {@code EPPHostAttr} is used to support the host attribute model.
 */
public class EPPHostAttr implements EPPCodecComponent {

	/**
	 * XML local name for {@code EPPHostAttr}.
	 */
	public static final String ELM_LOCALNAME = "hostAttr";

	/**
	 * XML Element Name of {@code EPPHostAttr} root element.
	 */
	public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML tag name for the {@code hostName} element.
	 */
	private final static java.lang.String ELM_HOST_NAME = "hostName";

	/**
	 * XML attribute name for the {@code hostAttr} element.
	 */
	private final static java.lang.String ELM_ADDRESS = "hostAddr";

	/**
	 * Host name
	 */
	private String name = null;

	/**
	 * Host addresses as a {@code Vector} of {@code EPPHostAddress} instances.
	 */
	private Vector addresses = null;

	/**
	 * {@code EPPHostAttr} default constructor.
	 */
	public EPPHostAttr() {
	}

	/**
	 * {@code EPPHostAttr} constructor the takes the host name.
	 *
	 * @param aName
	 *           the fully qualified name of a host
	 */
	public EPPHostAttr(String aName) {
		this.name = aName;
	}

	/**
	 * {@code EPPHostAttr} constructor the takes the host name and a
	 * {@code Vector} of host addresses.
	 *
	 * @param aName
	 *           the fully qualified name of a host
	 * @param aAddresses
	 *           {@code Vector} of {@code EPPHostAddress} instances
	 */
	public EPPHostAttr(String aName, Vector aAddresses) {
		this.name = aName;
		this.setAddresses(aAddresses);
	}

	/**
	 * Get the host name.
	 *
	 * @return Host Name
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Set the host name.
	 *
	 * @param aName
	 *           Host Name
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Gets the host addresses.
	 *
	 * @return Vector of {@code EPPHostAddress} instances if defined;
	 *         {@code null} otherwise.
	 */
	public Vector getAddresses() {
		return this.addresses;
	}

	/**
	 * Sets the host addresses.
	 *
	 * @param aAddresses
	 *           Vector of {@code EPPHostAddress} instances.
	 */
	public void setAddresses(Vector aAddresses) {
		this.addresses = aAddresses;

		// Set room tag of addresses
		if (this.addresses != null) {
			Iterator theIter = this.addresses.iterator();

			while (theIter.hasNext()) {
				EPPHostAddress currAddress = (EPPHostAddress) theIter.next();

				currAddress.setRootName(EPPDomainMapFactory.NS, EPPDomainMapFactory.NS_PREFIX + ":" + ELM_ADDRESS);
			}
		}
	}

	/**
	 * Clone {@code EPPHostAttr}.
	 *
	 * @return clone of {@code EPPHostAttr}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPHostAttr clone = (EPPHostAttr) super.clone();

		if (this.addresses != null) {
			clone.addresses = (Vector) this.addresses.clone();

			for (int i = 0; i < this.addresses.size(); i++) {
				clone.addresses.setElementAt(((EPPHostAddress) this.addresses.elementAt(i)).clone(), i);
			}
		}

		return clone;
	}

	/**
	 * Decode the EPPHostAttr attributes from the aElement DOM Element tree.
	 *
	 * @param aElement
	 *           - Root DOM Element to decode EPPHostAttr from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Host Name
		this.name = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_HOST_NAME);

		// Addresses
		this.addresses = EPPUtil.decodeCompVector(aElement, EPPDomainMapFactory.NS, ELM_ADDRESS, EPPHostAddress.class);

		if (this.addresses.size() == 0) {
			this.addresses = null;
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the EPPHostAttr instance.
	 *
	 * @param aDocument
	 *           - DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element - Root DOM Element representing the EPPHostAttr instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode EPPHostAttr instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		if (this.name == null) {
			throw new EPPEncodeException("EPPHostAttr: Host name is required");
		}

		Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_HOST_NAME);

		// Addresses
		EPPUtil.encodeCompVector(aDocument, root, this.addresses);

		return root;
	}

	/**
	 * implements a deep {@code EPPHostAttr} compare.
	 *
	 * @param aObject
	 *           {@code EPPHostAttr} instance to compare with
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPHostAttr)) {
			return false;
		}

		EPPHostAttr theComp = (EPPHostAttr) aObject;

		// Name
		if (!((this.name == null) ? (theComp.name == null) : this.name.equals(theComp.name))) {
			return false;
		}

		// Addresses
		if (!EPPUtil.equalVectors(this.addresses, theComp.addresses)) {
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPDomainMapFactory.NS;
	}
}
