/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodec;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Domain &lt;domain:transfer-data&gt; response to an
 * {@code EPPDomainTransferCmd}. When a &lt;transfer&gt; query command has been
 * processed successfully, the EPP &lt;resData&gt; element MUST contain a child
 * &lt;domain:trnData&gt; element that identifies the domain namespace and the
 * location of the domain schema. The &lt;domain:trnData&gt; element SHALL
 * contain the following child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;domain:name&gt; element that contains the fully qualified domain
 * name used in the query. Use {@code getName} and {@code setName} to get and
 * set the element.</li>
 * <li>A &lt;domain:reID&gt; element that contains the identifier of the client
 * that initiated the transfer request. Use {@code getRequestClient} and
 * {@code setRequestClient} to get and set the element.</li>
 * <li>A &lt;domain:acID&gt; element that contains the identifier of the client
 * that SHOULD respond to the transfer request. Use {@code getActionClient} and
 * {@code setActionClient} to get and set the element.</li>
 * <li>A &lt;domain:trStatus&gt; element that contains the state of the most
 * recent transfer request. Valid values are "PENDING", "APPROVED", "REJECTED",
 * "AUTO-APPROVED", "AUTO-REJECTED", and "CANCELLED". Use {@code getStatus} and
 * {@code setStatus} with the {@code EPPDomainTransferResp.STATUS_} constants to
 * get and set the element.</li>
 * <li>A &lt;domain:reDate&gt; element that contains the date and time that the
 * transfer was requested. Use {@code getRequestDate} and {@code setRequestDate}
 * to get and set the element.</li>
 * <li>A &lt;domain:acDate&gt; element that contains the date and time of a
 * required or completed response. For a PENDING request, the value identifies
 * the date and time by which a response is required before an automated
 * response action MUST be taken by the server. For all other status types, the
 * value identifies the date and time when the request was completed. Use
 * {@code getActionDate} and {@code setActionDate} to get and set the
 * element.</li>
 * <li>An OPTIONAL &lt;domain:exDate&gt; element that contains the end of the
 * domain's validity period if the &lt;transfer&gt; command caused or causes a
 * change in the validity period. Use {@code getExpirationDate} and
 * {@code setExpirationDate} to get and set the element.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.domain.EPPDomainTransferCmd
 */
public class EPPDomainTransferResp extends EPPResponse {
	/**
	 * XML local name for {@code EPPDomainTransferResp}.
	 */
	public static final String ELM_LOCALNAME = "trnData";

	/**
	 * XML Element Name of {@code EPPDomainTransferResp} root element.
	 */
	public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML tag name for the {@code name} attribute.
	 */
	private final static String ELM_DOMAIN_NAME = "name";

	/**
	 * XML tag name for the {@code requestClient} attribute.
	 */
	private final static String ELM_REQUEST_CLIENT = "reID";

	/**
	 * XML tag name for the {@code actionClient} attribute.
	 */
	private final static String ELM_ACTION_CLIENT = "acID";

	/**
	 * XML tag name for the {@code transferStatus} attribute.
	 */
	private final static String ELM_TRANSFER_STATUS = "trStatus";

	/**
	 * XML tag name for the {@code requestDate} attribute.
	 */
	private final static String ELM_REQUEST_DATE = "reDate";

	/**
	 * XML tag name for the {@code actionDate} attribute.
	 */
	private final static String ELM_ACTION_DATE = "acDate";

	/**
	 * XML tag name for the {@code expirationDate} attribute.
	 */
	private final static String ELM_EXPIRATION_DATE = "exDate";

	/**
	 * The fully qualified domain name.
	 */
	private String name = null;

	/**
	 * The identifier of the client that initiated the transfer request.
	 */
	private String requestClient = null;

	/**
	 * The identifier of the client that SHOULD respond to the transfer request.
	 */
	private String actionClient = null;

	/**
	 * The state of the most recent transfer request. This should be one of the
	 * {@code EPPDomainTransferResp.STATUS} constants.
	 */
	private String transferStatus = null;

	/**
	 * The date and time that the transfer was requested.
	 */
	private Date requestDate = null;

	/**
	 * The date and time of a required or completed response. For a
	 * STATUS_PENDING request, the value identifies the date and time by which a
	 * response is required before an automated response action MUST be taken by
	 * the server. For all other status types, the value identifies the date and
	 * time when the request was completed.
	 */
	private Date actionDate = null;

	/**
	 * An optional attribute that contains the end of the domain's validity
	 * period if the transfer command caused or causes a change in the validity
	 * period.
	 */
	private Date expirationDate = null;

	/**
	 * {@code EPPDomainTransferResp} default constructor. Must call required
	 * setter methods before invoking {@code encode}, which include:<br>
	 * <br>
	 *
	 * <ul>
	 * <li>name - {@code setName}</li>
	 * <li>request client - {@code setRequestClient}</li>
	 * <li>action client - {@code setActionClient}</li>
	 * <li>transfer status - {@code setTransferStatus}</li>
	 * <li>request date - {@code setReqeustDate}</li>
	 * <li>action date - {@code setActionDate}</li>
	 * <li>transaction id - {@code setTransId}</li>
	 * </ul>
	 */
	public EPPDomainTransferResp() {
		// Values set in attribute definition.
	}

	/**
	 * {@code EPPDomainTransferResp} which takes the name of domain. All other
	 * required attributes need to be set using the setter methods, which
	 * include:<br>
	 * <br>
	 *
	 * <ul>
	 * <li>request client - {@code setRequestClient}</li>
	 * <li>action client - {@code setActionClient}</li>
	 * <li>transfer status - {@code setTransferStatus}</li>
	 * <li>request date - {@code setReqeustDate}</li>
	 * <li>action date - {@code setActionDate}</li>
	 * </ul>
	 *
	 *
	 * @param aTransId
	 *           Transaction Id associated with response.
	 * @param aName
	 *           Domain name
	 */
	public EPPDomainTransferResp(EPPTransId aTransId, String aName) {
		super(aTransId);
		this.name = aName;
	}

	/**
	 * Gets the EPP response type associated with {@code EPPDomainTransferResp}.
	 *
	 * @return {@code EPPDomainTransferResp.ELM_NAME}
	 */
	@Override
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * {@code EPPDomainTransferResp}.
	 *
	 * @return {@code EPPDomainMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPDomainMapFactory.NS;
	}

	/**
	 * Validate the state of the {@code EPPDomainTransferResp} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            With invalid state
	 */
	void validateState() throws EPPCodecException {
		if (this.name == null) {
			throw new EPPCodecException("name required attribute is not set");
		}

		if (this.transferStatus == null) {
			throw new EPPCodecException("transferStatus required attribute is not set");
		}

		if (this.requestClient == null) {
			throw new EPPCodecException("requestClient required attribute is not set");
		}

		if (this.requestDate == null) {
			throw new EPPCodecException("requestDate required attribute is not set");
		}

		if (this.actionClient == null) {
			throw new EPPCodecException("actionClient required attribute is not set");
		}

		if (this.actionDate == null) {
			throw new EPPCodecException("actionDate required attribute is not set");
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPDomainTransferResp} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPDomainTransferResp}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPDomainTransferResp} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPDomainTransferResp.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);

		// Transfer Status
		EPPUtil.encodeString(aDocument, root, this.transferStatus, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_TRANSFER_STATUS);

		// Request Client
		EPPUtil.encodeString(aDocument, root, this.requestClient, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_REQUEST_CLIENT);

		// Request Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.requestDate, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_REQUEST_DATE);

		// Action Client
		EPPUtil.encodeString(aDocument, root, this.actionClient, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_ACTION_CLIENT);

		// Action Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.actionDate, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_ACTION_DATE);

		// Expiration Date
		if (this.expirationDate != null) {
			EPPUtil.encodeTimeInstant(aDocument, root, this.expirationDate, EPPDomainMapFactory.NS,
			      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_EXPIRATION_DATE);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPDomainTransferResp} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPDomainTransferResp} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);

		// Transfer Status
		this.transferStatus = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_TRANSFER_STATUS);

		// Request Client
		this.requestClient = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_REQUEST_CLIENT);

		// Request Date
		this.requestDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_REQUEST_DATE);

		// Action Client
		this.actionClient = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_ACTION_CLIENT);

		// Action Date
		this.actionDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_ACTION_DATE);

		// Expiration Date
		this.expirationDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_EXPIRATION_DATE);
	}

	/**
	 * Compare an instance of {@code EPPDomainTransferResp} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return DOCUMENT ME!
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPDomainTransferResp)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPDomainTransferResp theComp = (EPPDomainTransferResp) aObject;

		// Name
		if (!((this.name == null) ? (theComp.name == null) : this.name.equals(theComp.name))) {
			return false;
		}

		// Request Client
		if (!((this.requestClient == null) ? (theComp.requestClient == null) :
		      this.requestClient.equals(theComp.requestClient))) {
			return false;
		}

		// Action Client
		if (!((this.actionClient == null) ? (theComp.actionClient == null) :
		      this.actionClient.equals(theComp.actionClient))) {
			return false;
		}

		// Transfer Status
		if (!((this.transferStatus == null) ? (theComp.transferStatus == null) :
		      this.transferStatus.equals(theComp.transferStatus))) {
			return false;
		}

		// Request Date
		if (!((this.requestDate == null) ? (theComp.requestDate == null) :
		      this.requestDate.equals(theComp.requestDate))) {
			return false;
		}

		// Action Date
		if (!((this.actionDate == null) ? (theComp.actionDate == null) : this.actionDate.equals(theComp.actionDate))) {
			return false;
		}

		// Expiration Date
		if (!((this.expirationDate == null) ? (theComp.expirationDate == null) :
		      this.expirationDate.equals(theComp.expirationDate))) {
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPDomainTransferResp}.
	 *
	 * @return clone of {@code EPPDomainTransferResp}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPDomainTransferResp clone = (EPPDomainTransferResp) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the domain name
	 *
	 * @return Domain Name if set; {@code null} otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the domain name.
	 *
	 * @param aName
	 *           Domain Name
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Gets the identifier of the client that initiated the transfer request.
	 *
	 * @return The Request Client Id {@code String} if defined; {@code null}
	 *         otherwise.
	 */
	public String getRequestClient() {
		return this.requestClient;
	}

	/**
	 * Sets the identifier of the client that initiated the transfer request.
	 *
	 * @param aRequestClient
	 *           The Request Client Id {@code String}
	 */
	public void setRequestClient(String aRequestClient) {
		this.requestClient = aRequestClient;
	}

	/**
	 * Gets the identifier of the client that SHOULD respond to the transfer
	 * request.
	 *
	 * @return The Request Client Id {@code String} if defined; {@code null}
	 *         otherwise.
	 */
	public String getActionClient() {
		return this.actionClient;
	}

	/**
	 * Sets the identifier of the client that SHOULD respond to the transfer
	 * request.
	 *
	 * @param aActionClient
	 *           The Action Client Id {@code String}
	 */
	public void setActionClient(String aActionClient) {
		this.actionClient = aActionClient;
	}

	/**
	 * Gets the state of the most recent transfer request. This should be one of
	 * the {@code EPPResponse.TRANSFER} constants.
	 *
	 * @return The transfer status {@code String} if defined; {@code null}
	 *         otherwise.
	 */
	public String getTransferStatus() {
		return this.transferStatus;
	}

	/**
	 * Sets the state of the most recent transfer request. This should be one of
	 * the {@code EPPResponse.TRANSFER} constants.
	 *
	 * @param aTransferStatus
	 *           The transfer status String ({@code EPPResponse.TRANSFER})
	 */
	public void setTransferStatus(String aTransferStatus) {
		this.transferStatus = aTransferStatus;
	}

	/**
	 * Gets the date and time that the transfer was requested.
	 *
	 * @return The request date and time if defined; {@code null} otherwise.
	 */
	public Date getRequestDate() {
		return this.requestDate;
	}

	/**
	 * Sets the date and time that the transfer was requested.
	 *
	 * @param aRequestDate
	 *           The request date and time
	 */
	public void setRequestDate(Date aRequestDate) {
		this.requestDate = aRequestDate;
	}

	/**
	 * Gets the date and time of a required or completed response.
	 *
	 * @return The required or complete response data and time if defined;
	 *         {@code null} otherwise.
	 */
	public Date getActionDate() {
		return this.actionDate;
	}

	/**
	 * Sets the date and time of a required or completed response.
	 *
	 * @param aActionDate
	 *           The required or complete response data and time.
	 */
	public void setActionDate(Date aActionDate) {
		this.actionDate = aActionDate;
	}

	/**
	 * Gets the optional attribute that contains the end of the domain's validity
	 * period if the transfer command caused or causes a change in the validity
	 * period.
	 *
	 * @return Transfer expiration data and time if defined; {@code null}
	 *         otherwise.
	 */
	public Date getExpirationDate() {
		return this.expirationDate;
	}

	/**
	 * Sets the optional attribute that contains the end of the domain's validity
	 * period if the transfer command caused or causes a change in the validity
	 * period.
	 *
	 * @param aExpirationDate
	 *           Transfer expiration data and time.
	 */
	public void setExpirationDate(Date aExpirationDate) {
		this.expirationDate = aExpirationDate;
	}

}
