/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.validate.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPValidateKeyValue} is used to hold key, value pairs along with an
 * optional "contactType" attribute for use in {@link EPPValidateCheckCmd} and
 * {@link EPPValidateKeyValue}. {@code EPPValidateKeyValue} consists of a
 * required "key" and "value" attribute.
 *
 * @see com.verisign.epp.codec.validate.v02.EPPValidateCheckCmd
 * @see com.verisign.epp.codec.validate.v02.EPPValidateKeyValue
 */
public class EPPValidateKeyValue implements EPPCodecComponent {
	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPValidateKeyValue.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Constant for the key value local name
	 */
	public static final String ELM_LOCALNAME = "kv";

	/**
	 * Constant for the key value tag
	 */
	public static final String ELM_NAME = EPPValidateMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML key attribute
	 */
	private static final String ATTR_KEY = "key";

	/**
	 * XML value attribute
	 */
	private static final String ATTR_VALUE = "value";

	/**
	 * XML contactType attribute
	 */
	private static final String ATTR_CONTACT_TYPE = "contactType";

	/**
	 * Key attribute
	 */
	private String key;

	/**
	 * Value attribute
	 */
	private String value;

	/**
	 * The contact type.
	 */
	private String contactType;

	/**
	 * Default constructor for {@code EPPValidateKeyValue}.
	 */
	public EPPValidateKeyValue() {
	}

	/**
	 * Constructor for {@code EPPValidateKeyValue} that includes the required
	 * attributes of the key and the value.
	 *
	 * @param aKey
	 *           Key associated with the value
	 * @param aValue
	 *           Value associated with the key
	 */
	public EPPValidateKeyValue(String aKey, String aValue) {
		this.key = aKey;
		this.value = aValue;
	}

	/**
	 * Constructor for {@code EPPValidateKeyValue} that includes all of the
	 * attributes.
	 *
	 * @param aKey
	 *           Key associated with the value
	 * @param aValue
	 *           Value associated with the key
	 * @param aContactType
	 *           Contact type associated with this key value pair.
	 */
	public EPPValidateKeyValue(String aKey, String aValue, String aContactType) {
		this.key = aKey;
		this.value = aValue;
		this.contactType = aContactType;
	}

	/**
	 * Gets the key.
	 *
	 * @return The key if defined;{@code null} otherwise.
	 */
	public String getKey() {
		return this.key;
	}

	/**
	 * Sets the key.
	 *
	 * @param aKey
	 *           The key
	 */
	public void setKey(String aKey) {
		this.key = aKey;
	}

	/**
	 * Gets the value.
	 *
	 * @return The value if defined;{@code null} otherwise.
	 */
	public String getValue() {
		return this.value;
	}

	/**
	 * Sets the value.
	 *
	 * @param aValue
	 *           The value
	 */
	public void setValue(String aValue) {
		this.value = aValue;
	}

	/**
	 * Is the contact type defined?
	 *
	 * @return {@code true} if the contact type is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasContactType() {
		return (this.contactType != null ? true : false);
	}

	/**
	 * Gets the contact type.
	 *
	 * @return The contact type if defined;{@code null} otherwise.
	 */
	public String getContactType() {
		return this.contactType;
	}

	/**
	 * Sets the contact type.
	 *
	 * @param aType
	 *           The contact type
	 */
	public void setContactType(String aContactType) {
		this.contactType = aContactType;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPValidateKeyValue} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPValidateKeyValue} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPValidateKeyValue} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Check required attributes
		if (this.key == null) {
			throw new EPPEncodeException("Undefined key in EPPValidateKeyValue");
		}
		if (this.value == null) {
			throw new EPPEncodeException("Undefined value in EPPValidateKeyValue");
		}

		Element root = aDocument.createElementNS(EPPValidateMapFactory.NS, ELM_NAME);

		// Key
		root.setAttribute(ATTR_KEY, this.key);

		// Value
		root.setAttribute(ATTR_VALUE, this.value);

		// Contact Type
		if (this.hasContactType()) {
			root.setAttribute(ATTR_CONTACT_TYPE, this.contactType);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPValidateKeyValue} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPValidateKeyValue} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Key
		this.key = EPPUtil.decodeStringAttr(aElement, ATTR_KEY);

		// Value
		this.value = EPPUtil.decodeStringAttr(aElement, ATTR_VALUE);

		// Contact Type
		this.contactType = EPPUtil.decodeStringAttr(aElement, ATTR_CONTACT_TYPE);
	}

	/**
	 * Compare an instance of {@code EPPValidateKeyValue} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPValidateKeyValue)) {
			return false;
		}

		EPPValidateKeyValue other = (EPPValidateKeyValue) aObject;

		// Key
		if (!EqualityUtil.equals(this.key, other.key)) {
			cat.error("EPPValidateKeyValue.equals(): key not equal");
			return false;
		}

		// Value
		if (!EqualityUtil.equals(this.value, other.value)) {
			cat.error("EPPValidateKeyValue.equals(): value not equal");
			return false;
		}

		// Contact Type
		if (!EqualityUtil.equals(this.contactType, other.contactType)) {
			cat.error("EPPValidateKeyValue.equals(): contactType not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPValidateKeyValue}.
	 *
	 * @return clone of {@code EPPValidateKeyValue}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPValidateKeyValue clone = null;

		clone = (EPPValidateKeyValue) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPValidateMapFactory.NS;
	}

}
