/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.validate.v02;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents a &lt;validate:contact&gt; element in the Validate Check Command.
 */
public class EPPValidateContact implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPValidateContact.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPValidateContact}.
	 */
	public static final String ELM_LOCALNAME = "contact";

	/**
	 * XML root tag for {@code EPPValidateContact}.
	 */
	public static final String ELM_NAME = EPPValidateMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the contact identifier element
	 */
	private static final String ELM_ID = "id";

	/**
	 * XML local name for the OPTIONAL voice number ({@code voice}) element.
	 */
	private final static String ELM_VOICE = "voice";

	/**
	 * XML local name for the OPTIONAL fax number ({@code fax}) element.
	 */
	private final static String ELM_FAX = "fax";

	/**
	 * XML local name for the email ({@code email}) element.
	 */
	private final static String ELM_EMAIL = "email";

	/**
	 * XML extension "x" attribute value for voice or fax elements.
	 */
	private final static String ATTR_EXT = "x";

	/**
	 * XML extension "contactType" attribute value.
	 */
	private final static String ATTR_CONTACT_TYPE = "contactType";

	/**
	 * XML extension "tld" attribute value.
	 */
	private final static String ATTR_TLD = "tld";

	/**
	 * The contact identifier.
	 */
	private String id;

	/**
	 * The contact type.
	 */
	private String contactType;

	/**
	 * The TLD.
	 */
	private String tld;

	/**
	 * One or two postal information elements, represented by the
	 * {@link EPPValidatePostalDefinition} class.
	 */
	private List<EPPValidatePostalDefinition> postalInfo = new ArrayList<EPPValidatePostalDefinition>();

	/**
	 * OPTIONAL Voice number
	 */
	private String voice;

	/**
	 * OPTIONAL Voice Extension
	 */
	private String voiceExt;

	/**
	 * OPTIONAL FAX number
	 */
	private String fax;

	/**
	 * OPTIONAL FAX Extension
	 */
	private String faxExt;

	/**
	 * OPTIONAL Email Address
	 */
	private String email;

	/**
	 * OPTIONAL authInfo value
	 */
	private EPPValidateAuthInfo authInfo;

	/**
	 * Optional list of key value pairs used to define additional contact
	 * attributes. The {@link EPPValidateKeyValue} {@code contactType} attribute
	 * should not be set in each key value pair.
	 */
	private List<EPPValidateKeyValue> keyValues = new ArrayList<EPPValidateKeyValue>();

	/**
	 * {@code EPPValidateContact} default constructor.
	 */
	public EPPValidateContact() {
	}

	/**
	 * {@code EPPValidateContact} default constructor that takes the required
	 * attributes.
	 *
	 * @param aId
	 *           Contact identifier
	 * @param aContactType
	 *           Contact type
	 * @param aTld
	 *           TLD
	 */
	public EPPValidateContact(String aId, String aContactType, String aTld) {
		this.id = aId;
		this.contactType = aContactType;
		this.tld = aTld;
	}

	/**
	 * Gets the contact identifier.
	 *
	 * @return The contact identifier if defined;{@code null} otherwise.
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * Sets the contact identifier.
	 *
	 * @param aId
	 *           The contact identifier
	 */
	public void setId(String aId) {
		this.id = aId;
	}

	/**
	 * Gets the contact type.
	 *
	 * @return The contact type if defined;{@code null} otherwise.
	 */
	public String getContactType() {
		return this.contactType;
	}

	/**
	 * Sets the contact type.
	 *
	 * @param aType
	 *           The contact type
	 */
	public void setContactType(String aContactType) {
		this.contactType = aContactType;
	}

	/**
	 * Gets the TLD.
	 *
	 * @return The TLD if defined;{@code null} otherwise.
	 */
	public String getTld() {
		return this.tld;
	}

	/**
	 * Sets the TLD.
	 *
	 * @param aTld
	 *           The TLD
	 */
	public void setTld(String aTld) {
		this.tld = aTld;
	}

	/**
	 * Is there any postal information set?
	 *
	 * @return {@code true} if there is at least one
	 *         {@link EPPValidatePostalDefinition} set in the postal information;
	 *         {@code false} otherwise.
	 */
	public boolean hasPostalInfo() {
		if (this.postalInfo != null && !this.postalInfo.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a postal definition to the postal information for the contact.
	 *
	 * @param aPostalInfo
	 *           Postal definition to add to the postal information.
	 */
	public void addPostalInfo(EPPValidatePostalDefinition aPostalInfo) {
		this.postalInfo.add(aPostalInfo);
	}

	/**
	 * Gets the postal information for the contact. There can be one or two
	 * {@link EPPValidatePostalDefinition} objects in the postal information
	 * list.
	 *
	 * @return Postal information for the contact
	 */
	public List<EPPValidatePostalDefinition> getPostalInfo() {
		return this.postalInfo;
	}

	/**
	 * Sets the postal information for the contact. There can be one or two
	 * {@link EPPValidatePostalDefinition} objects in the postal information
	 * list.
	 *
	 * @param aPostalInfo
	 *           Postal information for the contact.
	 */
	public void setPostalInfo(List<EPPValidatePostalDefinition> aPostalInfo) {
		if (aPostalInfo == null) {
			this.postalInfo = new ArrayList<EPPValidatePostalDefinition>();
		}
		else {
			this.postalInfo = aPostalInfo;
		}
	}

	/**
	 * Is the contact voice value been defined?
	 *
	 * @return <code>true</code> if the contact voice value is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasVoice() {
		return (this.voice != null ? true : false);
	}

	/**
	 * Gets the contact voice value.
	 * 
	 * @return the contact voice value if defined; {@code null} otherwise.
	 */
	public String getVoice() {
		return this.voice;
	}

	/**
	 * Sets the contact voice value.
	 * 
	 * @param aVoice
	 *           the contact voice value.
	 */
	public void setVoice(String aVoice) {
		this.voice = aVoice;
	}

	/**
	 * Is the contact voice extension value defined?
	 *
	 * @return <code>true</code> if the contact voice extension value is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasVoiceExt() {
		return (this.voiceExt != null ? true : false);
	}

	/**
	 * Gets the contact voice extension value.
	 * 
	 * @return the contact voice extension value if defined; {@code null}
	 *         otherwise.
	 */
	public String getVoiceExt() {
		return this.voiceExt;
	}

	/**
	 * Sets the contact voice extension value.
	 * 
	 * @param aVoiceExt
	 *           the contact voice extension value
	 */
	public void setVoiceExt(String aVoiceExt) {
		this.voiceExt = aVoiceExt;
	}

	/**
	 * Is the contact fax value been defined?
	 *
	 * @return <code>true</code> if the contact fax value is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasFax() {
		return (this.fax != null ? true : false);
	}

	/**
	 * Gets the contact fax value.
	 * 
	 * @return the contact fax value if defined; {@code null} otherwise.
	 */
	public String getFax() {
		return this.fax;
	}

	/**
	 * Sets the contact fax value.
	 * 
	 * @param aFax
	 *           the contact fax value.
	 */
	public void setFax(String aFax) {
		this.fax = aFax;
	}

	/**
	 * Is the contact fax extension value defined?
	 *
	 * @return <code>true</code> if the contact fax extension value is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasFaxExt() {
		return (this.faxExt != null ? true : false);
	}

	/**
	 * Gets the contact fax extension value.
	 * 
	 * @return the contact fax extension value if defined; {@code null}
	 *         otherwise.
	 */
	public String getFaxExt() {
		return this.faxExt;
	}

	/**
	 * Sets the contact fax extension value.
	 * 
	 * @param aFaxExt
	 *           the contact fax extension value
	 */
	public void setFaxExt(String aFaxExt) {
		this.faxExt = aFaxExt;
	}

	/**
	 * Is the email defined?
	 *
	 * @return <code>true</code> if the email is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasEmail() {
		return (this.email != null ? true : false);
	}

	/**
	 * Gets the contact email value.
	 * 
	 * @return the email if defined; {@code null} otherwise.
	 */
	public String getEmail() {
		return this.email;
	}

	/**
	 * Sets the contact email value.
	 * 
	 * @param aEmail
	 *           the email to set
	 */
	public void setEmail(String aEmail) {
		this.email = aEmail;
	}

	/**
	 * Is the authorization info defined?
	 *
	 * @return <code>true</code> if the authorization info is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasAuthInfo() {
		return (this.authInfo != null ? true : false);
	}

	/**
	 * Gets the authorization info value.
	 * 
	 * @return the authorization info value if defined; {@code null} otherwise.
	 */
	public EPPValidateAuthInfo getAuthInfo() {
		return this.authInfo;
	}

	/**
	 * Sets the authorization info value.
	 * 
	 * @param aAuthInfo
	 *           The authorization info value
	 */
	public void setAuthInfo(EPPValidateAuthInfo aAuthInfo) {
		this.authInfo = aAuthInfo;
	}

	/**
	 * Adds a key value pair to the list of key value pairs. The key value pair
	 * is used to defined additional contact attributes. The key value pair
	 * should not include the {@code contactType} attribute.
	 *
	 * @param aKeyValue
	 *           Key value pair to add to the list
	 */
	public void addKeyValue(EPPValidateKeyValue aKeyValue) {
		this.keyValues.add(aKeyValue);
	}

	/**
	 * Sets the full list of key value pairs. A key value pair is used to define
	 * additional contact attributes.
	 *
	 * @param aKeyValues
	 *           List of {@code EPPValidateKeyValue} instances.
	 */
	public void setKeyValues(List<EPPValidateKeyValue> aKeyValues) {
		if (aKeyValues == null) {
			this.keyValues = new ArrayList<EPPValidateKeyValue>();
		}
		else {
			this.keyValues = aKeyValues;
		}
	}

	/**
	 * Gets the full list of key value pairs. A key value pair is used to define
	 * additional contact attributes.
	 *
	 *
	 * @return List of {@code EPPValidateKeyValue} instances.
	 */
	public List<EPPValidateKeyValue> getKeyValues() {
		return this.keyValues;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPValidateContact} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPValidateCheckResult} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPValidateCheckResult} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		Element theElm = null;
		Text theVal = null;

		// Check required attributes
		if (this.id == null) {
			throw new EPPEncodeException("Undefined id in EPPValidateContact");
		}
		if (this.contactType == null) {
			throw new EPPEncodeException("Undefined contactType in EPPValidateContact");
		}
		if (this.tld == null) {
			throw new EPPEncodeException("Undefined tld in EPPValidateContact");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPValidateMapFactory.NS, ELM_NAME);

		// Contact Type
		root.setAttribute(ATTR_CONTACT_TYPE, this.contactType);

		// TLD
		root.setAttribute(ATTR_TLD, this.tld);

		// Contact Identifier
		EPPUtil.encodeString(aDocument, root, this.id, EPPValidateMapFactory.NS,
		      EPPValidateMapFactory.NS_PREFIX + ":" + ELM_ID);

		// Postal Information
		EPPUtil.encodeCompList(aDocument, root, this.postalInfo);

		// Voice and Voice Extension
		if (this.voice != null) {
			theElm = aDocument.createElementNS(EPPValidateMapFactory.NS,
			      EPPValidateMapFactory.NS_PREFIX + ":" + ELM_VOICE);
			theVal = aDocument.createTextNode(this.voice);
			theElm.appendChild(theVal);

			if (this.voiceExt != null) {
				theElm.setAttribute(ATTR_EXT, this.voiceExt);
			}

			root.appendChild(theElm);
		}

		// Fax and Fax Extension
		if (this.fax != null) {
			theElm = aDocument.createElementNS(EPPValidateMapFactory.NS, EPPValidateMapFactory.NS_PREFIX + ":" + ELM_FAX);
			theVal = aDocument.createTextNode(this.fax);
			theElm.appendChild(theVal);

			if (this.faxExt != null) {
				theElm.setAttribute(ATTR_EXT, this.faxExt);
			}

			root.appendChild(theElm);
		}

		// Email
		EPPUtil.encodeString(aDocument, root, this.email, EPPValidateMapFactory.NS,
		      EPPValidateMapFactory.NS_PREFIX + ":" + ELM_EMAIL);

		// Auth Info
		EPPUtil.encodeComp(aDocument, root, this.authInfo);

		// Key Values
		EPPUtil.encodeCompList(aDocument, root, this.keyValues);

		return root;
	}

	/**
	 * Decode the {@code EPPValidateContact} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPValidateCheckResult} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		Element theElm = null;

		// Contact Identifier
		this.id = EPPUtil.decodeString(aElement, EPPValidateMapFactory.NS, ELM_ID);

		// Contact Type
		this.contactType = EPPUtil.decodeStringAttr(aElement, ATTR_CONTACT_TYPE);

		// TLD
		this.tld = EPPUtil.decodeStringAttr(aElement, ATTR_TLD);

		// Postal Information
		this.postalInfo = EPPUtil.decodeCompList(aElement, EPPValidateMapFactory.NS, EPPValidatePostalDefinition.ELM_NAME,
		      EPPValidatePostalDefinition.class);

		// Voice and Voice Extension
		theElm = EPPUtil.getElementByTagNameNS(aElement, EPPValidateMapFactory.NS, ELM_VOICE);
		if (theElm != null) {
			this.voice = EPPUtil.getTextContent(theElm);
			this.voiceExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
		}
		else {
			this.voice = null;
			this.voiceExt = null;
		}

		// Fax and Fax Extension
		theElm = EPPUtil.getElementByTagNameNS(aElement, EPPValidateMapFactory.NS, ELM_FAX);
		if (theElm != null) {
			this.fax = EPPUtil.getTextContent(theElm);
			this.faxExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
		}
		else {
			this.fax = null;
			this.faxExt = null;
		}

		// Email
		this.email = EPPUtil.decodeString(aElement, EPPValidateMapFactory.NS, ELM_EMAIL);

		// Auth Info
		this.authInfo = (EPPValidateAuthInfo) EPPUtil.decodeComp(aElement, EPPValidateMapFactory.NS,
		      EPPValidateAuthInfo.ELM_NAME, EPPValidateAuthInfo.class);

		// Key Values
		this.keyValues = EPPUtil.decodeCompList(aElement, EPPValidateMapFactory.NS, EPPValidateKeyValue.ELM_NAME,
		      EPPValidateKeyValue.class);
	}

	/**
	 * Clone {@code EPPValidateContact}.
	 *
	 * @return clone of {@code EPPValidateContact}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPValidateContact clone = (EPPValidateContact) super.clone();

		if (this.hasPostalInfo()) {
			clone.postalInfo = (List) ((ArrayList) this.postalInfo).clone();
		}

		// Auth Info
		if (this.authInfo != null) {
			clone.authInfo = (EPPValidateAuthInfo) this.authInfo.clone();
		}

		// Key Values
		clone.keyValues = new ArrayList<EPPValidateKeyValue>(this.keyValues.size());
		for (EPPValidateKeyValue item : this.keyValues) {
			clone.keyValues.add((EPPValidateKeyValue) item.clone());
		}

		return clone;
	}

	/**
	 * Gets the EPP response type associated with {@code EPPValidateContact}.
	 *
	 * @return {@code EPPValidateContact.ELM_NAME}
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command namespace associated with {@code EPPValidateContact}.
	 *
	 * @return {@code EPPValidateMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPValidateMapFactory.NS;
	}

	/**
	 * Compare an instance of {@code EPPValidateContact} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPValidateContact)) {
			return false;
		}

		EPPValidateContact other = (EPPValidateContact) aObject;

		// Contact Identifier
		if (!EqualityUtil.equals(this.id, other.id)) {
			cat.error("EPPValidateContact.equals(): id not equal");
			return false;
		}

		// Contact Type
		if (!EqualityUtil.equals(this.contactType, other.contactType)) {
			cat.error("EPPValidateContact.equals(): contactType not equal");
			return false;
		}

		// TLD
		if (!EqualityUtil.equals(this.tld, other.tld)) {
			cat.error("EPPValidateContact.equals(): tld not equal");
			return false;
		}

		// Postal Information
		if (!EPPUtil.equalLists(this.postalInfo, other.postalInfo)) {
			cat.error("EPPValidateContact.equals(): postalInfo not equal");
			return false;
		}

		// Voice
		if (!EqualityUtil.equals(this.voice, other.voice)) {
			cat.error("EPPValidateContact.equals(): voice not equal");
			return false;
		}

		// Voice Ext
		if (!EqualityUtil.equals(this.voiceExt, other.voiceExt)) {
			cat.error("EPPValidateContact.equals(): voiceExt not equal");
			return false;
		}

		// Fax
		if (!EqualityUtil.equals(this.fax, other.fax)) {
			cat.error("EPPValidateContact.equals(): fax not equal");
			return false;
		}

		// Fax Ext
		if (!EqualityUtil.equals(this.faxExt, other.faxExt)) {
			cat.error("EPPValidateContact.equals(): faxExt not equal, this.faxExt = " + this.faxExt + ", other.faxExt = "
			      + other.faxExt);
			return false;
		}

		// Email
		if (!EqualityUtil.equals(this.email, other.email)) {
			cat.error("EPPValidateContact.equals(): email not equal");
			return false;
		}

		// Auth Info
		if (!EqualityUtil.equals(this.authInfo, other.authInfo)) {
			cat.error("EPPValidateContact.equals(): authInfo not equal");
			return false;
		}

		// Key Values
		if (!EqualityUtil.equals(this.keyValues, other.keyValues)) {
			cat.error("EPPValidateContact.equals(): keyValues not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

}
