/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Defines the miniumum and maximum numbers of contacts by contact type. The
 * contact type is defined with the required "type" attribute with the possible
 * values of "admin", "tech", and "billing". The &lt;registry:contact&gt;
 * element contains the following child elements<br>
 * <br>
 * <ul>
 * <li>&lt;registry:min&gt; - The minimum number of contacts for the contact
 * type. Use {@link #getMin()} and {@link #setMin(Integer)} to get and set this
 * element.</li>
 * <li>&lt;registry:max&gt; - The OPTIONAL maximum number of contacts for the
 * contact type. If this element is not defined, the maximum number is
 * unbounded. Use {@link #getMax()} and {@link #setMax(Integer)} to get and set
 * this element.</li>
 * </ul>
 * 
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryDomain
 */
public class EPPRegistryDomainContact extends EPPRegistryAbstractMinMax {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryDomainContact.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryDomainContact}.
	 */
	public static final String ELM_LOCALNAME = "contact";

	/**
	 * XML root tag for {@code EPPRegistryDomainContact}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML attribute name for the {@code type} attribute.
	 */
	public static final String ATTR_TYPE = "type";

	/**
	 * Possible values for the {@code sharePolicy} attribute.
	 */
	public static enum Type {

		/**
		 * administrative contact
		 */
		admin,

		/**
		 * technical contact
		 */
		tech,

		/**
		 * billing contact
		 */
		billing
	}

	/**
	 * contact type
	 */
	private Type type = null;

	/**
	 * Default constructor of {@code EPPRegistryDomainContact}. All attributes
	 * are set to null. Must call {@link #setType(String)} and
	 * {@link #setMin(Integer)} before calling {@link #encode(Document)}.
	 */
	public EPPRegistryDomainContact() {
	}

	/**
	 * Creates a new {@code EPPRegistryDomainContact} object that takes the types
	 * and both the minimum and maximum values as {@code Integer}'s.
	 * 
	 * @param aType
	 *           type of contact
	 * @param aMin
	 *           minimum number of contact entries.
	 * @param aMax
	 *           maximum number of contact entries.
	 */
	public EPPRegistryDomainContact(Type aType, Integer aMin, Integer aMax) {
		super(aMin, aMax);
		this.type = aType;
	}

	/**
	 * Creates a new {@code EPPRegistryDomainContact} object that takes the types
	 * and both the minimum and maximum values as {@code int}'s.
	 * 
	 * @param aType
	 *           type of contact
	 * @param aMin
	 *           minimum number of contact entries.
	 * @param aMax
	 *           maximum number of contact entries.
	 */
	public EPPRegistryDomainContact(Type aType, int aMin, int aMax) {
		this(aType, Integer.valueOf(aMin), Integer.valueOf(aMax));
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryDomainContact} instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryDomainContact} instance.
	 * 
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryDomainContact}
	 *               instance.
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {
		this.validateState();

		Element root = super.encode(aDocument);

		// Type
		root.setAttribute(ATTR_TYPE, type.toString());

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryDomainContact} attributes from the aElement
	 * DOM Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryDomainContact}
	 *           from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		super.decode(aElement);
		
		// Type
		String theTypeStr = aElement.getAttribute(ATTR_TYPE);
		if (theTypeStr == null) {
			this.type = null;
		}
		else {
			this.type = Type.valueOf(theTypeStr);
		}
		
	}

	/**
	 * Clone {@code EPPRegistryDomainContact}.
	 * 
	 * @return clone of {@code EPPRegistryDomainContact}
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		return (EPPRegistryDomainContact) super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryDomainContact} compare.
	 * 
	 * @param aObject
	 *           {@code EPPRegistryDomainContact} instance to compare with
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!super.equals(aObject)) {
			return false;
		}

		if (!(aObject instanceof EPPRegistryDomainContact)) {
			return false;
		}

		EPPRegistryDomainContact theComp = (EPPRegistryDomainContact) aObject;

		if (!EqualityUtil.equals(this.type, theComp.type)) {
			cat.error("EPPRegistryDomainContact.equals(): type not equal");
			return false;
		}
		
		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 * 
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the type of contact.
	 * 
	 * @return Type of contact if defined; {@code null} otherwise.
	 */
	public Type getType() {
		return type;
	}

	/**
	 * Sets the type of contact.
	 * 
	 * @param aType
	 *           The type of contact
	 */
	public void setType(Type aType) {
		this.type = aType;
	}

	/**
	 * Gets the root element local name.
	 * 
	 * @return Root element local name.
	 */
	@Override
	protected String getRootName() {
		return ELM_LOCALNAME;
	}

	/**
	 * Gets the minimum element local name.
	 * 
	 * @return Minimum element local name.
	 */
	@Override
	protected String getElmMin() {
		return ELM_MIN;
	}

	/**
	 * Gets the maximum element local name.
	 * 
	 * @return Maximum element local name.
	 */
	@Override
	protected String getElmMax() {
		return ELM_MAX;
	}

	/**
	 * Gets the {@link Logger} to use.
	 * 
	 * @return {@link Logger} instance to use for logging.
	 */
	@Override
	protected Logger getLogger() {
		return cat;
	}

	/**
	 * Validate the state of the {@code EPPRegistryDomainContact} instance. A
	 * valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the {@code EPPCodecException} will contain a
	 * description of the error.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	protected void validateState() throws EPPEncodeException {
		// Type
		if (this.type == null) {
			throw new EPPEncodeException("Required type attribute is null");
		}

		super.validateState();
	}

}
