/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Abstract base class for defining a complex element that includes {@code min} and {@code max}
 * sub-elements.  
 */
public abstract class EPPRegistryAbstractMinMax implements EPPCodecComponent {


	/**
	 * Possible XML Element Name of {@code min} attribute.
	 */
	protected static final String ELM_MIN_LENGTH = "minLength";

	/**
	 * Possible XML Element Name of {@code max} attribute.
	 */
	protected static final String ELM_MAX_LENGTH = "maxLength";

	
	/**
	 * Possible XML Element Name of {@code min} attribute.
	 */
	protected static final String ELM_MIN = "min";

	/**
	 * Possible XML Element Name of {@code max} attribute.
	 */
	protected static final String ELM_MAX = "max";
	
	/**
	 * Minimal value
	 */
	private Integer min = null;

	/**
	 * Maximum value
	 */
	private Integer max = null;

	/**
	 * Gets the root element local name.
	 *
	 * @return Root element local name.
	 */
	protected abstract String getRootName();

	/**
	 * Gets the minimum element local name.
	 *
	 * @return Minimum element local name.
	 */
	protected abstract String getElmMin();

	/**
	 * Gets the maximum element local name.
	 *
	 * @return Maximum element local name.
	 */
	protected abstract String getElmMax();
	
	/**
	 * Gets the {@link Logger} to use.
	 * @return {@link Logger} instance to use for logging.
	 */
	protected abstract Logger getLogger();

	/**
	 * Default {@code EPPRegistryAbstractMinMax} constructor.
	 */
	public EPPRegistryAbstractMinMax() {
		super();
	}

	/**
	 * {@code EPPRegistryAbstractMinMax} that takes both the minimum and maximum
	 * values as {@code Integer}'s.
	 *
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryAbstractMinMax(Integer aMin, Integer aMax) {
		this();
		this.min = aMin;
		this.max = aMax;
	}

	/**
	 * {@code EPPRegistryAbstractMinMax} that takes both the minimum and maximum
	 * values as {@code int}'s.
	 *
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryAbstractMinMax(int aMin, int aMax) {
		this(Integer.valueOf(aMin), Integer.valueOf(aMax));
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryAbstractMinMax} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryAbstractMinMax} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryAbstractMinMax} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		validateState();

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + getRootName());

		// Min
		EPPUtil.encodeString(aDocument, root, this.min.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + getElmMin());

		// Max
		if (this.max != null) {
			EPPUtil.encodeString(aDocument, root, this.max.toString(), EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + getElmMax());
		}

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryAbstractMinMax} attributes from the aElement
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryAbstractMinMax} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		this.min = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, getElmMin());
		this.max = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, getElmMax());
	}

	/**
	 * Validate the state of the {@code EPPRegistryAbstractMinMax} instance. A
	 * valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the {@code EPPCodecException} will contain a description of
	 * the error. 
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	protected void validateState() throws EPPEncodeException {
		// Min
		if (this.min == null || this.min.intValue() < 0) {
			throw new EPPEncodeException("Invalid state on " + getClass().getName()
			      + ".encode: min is required and should be greater than or equal to 0");
		}

		// Max
		if (this.max != null && this.max.intValue() < this.min.intValue()) {
			throw new EPPEncodeException(
			      "Invalid state on " + getClass().getName() + ".encode: max is required and should be greater than min");
		}
	}

	/**
	 * Clone {@code EPPRegistryAbstractMinMax}.
	 *
	 * @return clone of {@code EPPRegistryAbstractMinMax}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryAbstractMinMax} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryAbstractMinMax} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryAbstractMinMax)) {
			return false;
		}

		EPPRegistryAbstractMinMax theComp = (EPPRegistryAbstractMinMax) aObject;

		// Min
		if (!EqualityUtil.equals(this.min, theComp.min)) {
			getLogger().error("EPPRegistryAbstractMinMax.equals(): min not equal");
			return false;
		}

		// Max
		if (!EqualityUtil.equals(this.max, theComp.max)) {
			getLogger().error("EPPRegistryAbstractMinMax.equals(): max not equal");
			return false;
		}

		return true;
	}

	/**
	 * Gets the minimum value.
	 *
	 * @return Minimum value if defined; {@code null} otherwise.
	 */
	public Integer getMin() {
		return this.min;
	}

	/**
	 * Sets the minimum value.
	 *
	 * @param aMin
	 *           Minimum value
	 */
	public void setMin(Integer aMin) {
		this.min = aMin;
	}

	/**
	 * Gets the maximum value.
	 *
	 * @return Maximum value if defined; {@code null} otherwise.
	 */
	public Integer getMax() {
		return this.max;
	}

	/**
	 * Sets the maximum value.
	 *
	 * @param aMax
	 *           Maximum value
	 */
	public void setMax(Integer aMax) {
		this.max = aMax;
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

}