/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.util.EPPCatFactory;

/**
 * Represents a general element that supports min and max length sub-elements. The
 * actual XML elements are configurable include:
 * <ul>
 * <li>root element name - The root element name must be defined.</li>
 * <li>min sub-element name - The min sub-element name can be overridden with
 * the default set to {@link EPPRegistryAbstractMinMax#ELM_MIN_LENGTH}.</li>
 * <li>max sub-element name - The max sub-element name can be overridden with
 * the default set to {@link EPPRegistryAbstractMinMax#ELM_MAX_LENGTH}.</li>
 * </ul>
 */
public class EPPRegistryMinMaxLength extends EPPRegistryAbstractMinMax {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryMinMaxLength.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name that must be set prior to calling
	 * {@link #encode(org.w3c.dom.Document)}.
	 */
	private String elmRootLocalName;

	/**
	 * XML local name for the {@code min} attribute with the default of
	 * {@link EPPRegistryAbstractMinMax#ELM_MIN_LENGTH}.
	 */
	private String elmMinLocalName = EPPRegistryAbstractMinMax.ELM_MIN_LENGTH;

	/**
	 * XML local name for the {@code max} with the default of
	 * {@link EPPRegistryAbstractMinMax#ELM_MAX_LENGTH}.
	 */
	private String elmMaxLocalName = EPPRegistryAbstractMinMax.ELM_MAX_LENGTH;

	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object.
	 */
	public EPPRegistryMinMaxLength() {
	}

	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object that takes both the
	 * minimum and maximum values as {@code Integer}'s.
	 *
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryMinMaxLength(Integer aMin, Integer aMax) {
		super(aMin, aMax);
	}

	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object that takes both the
	 * minimum and maximum values as {@code int}'s.
	 *
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryMinMaxLength(int min, int max) {
		this(Integer.valueOf(min), Integer.valueOf(max));
	}
	
	
	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object that takes both the
	 * minimum and maximum values as {@code Integer}'s.
	 *
	 * @param aRootLocalName
	 *           The XML root element local name to use.
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryMinMaxLength(String aRootLocalName, Integer aMin, Integer aMax) {
		super(aMin, aMax);
		this.elmRootLocalName = aRootLocalName;
	}

	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object that takes both the
	 * minimum and maximum values as {@code int}'s.
	 *
	 * @param aRootLocalName
	 *           The XML root element local name to use.
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryMinMaxLength(String aRootLocalName, int min, int max) {
		this(aRootLocalName, Integer.valueOf(min), Integer.valueOf(max));
	}

	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object that takes both the
	 * minimum and maximum values as {@code Integer}'s.
	 *
	 * @param aRootLocalName
	 *           The XML root element local name to use.
	 * @param aMinLocalName
	 *           The XML local name used for the {@code min} attribute.
	 * @param aMaxLocalName
	 *           The XML local name used for the {@code max} attribute.
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryMinMaxLength(String aRootLocalName, String aMinLocalName, String aMaxLocalName, Integer aMin,
	      Integer aMax) {
		super(aMin, aMax);
		this.elmRootLocalName = aRootLocalName;
		this.elmMinLocalName = aMinLocalName;
		this.elmMaxLocalName = aMaxLocalName;
	}

	/**
	 * Creates a new {@code EPPRegistryMinMaxLength} object that takes both the
	 * minimum and maximum values as {@code int}'s.
	 *
	 * @param aRootLocalName
	 *           The XML root element local name to use.
	 * @param aMinLocalName
	 *           The XML local name used for the {@code min} attribute.
	 * @param aMaxLocalName
	 *           The XML local name used for the {@code max} attribute.
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 */
	public EPPRegistryMinMaxLength(String aRootLocalName, String aMinLocalName, String aMaxLocalName, int min, int max) {
		this(aRootLocalName, aMinLocalName, aMaxLocalName, Integer.valueOf(min), Integer.valueOf(max));
	}

	/**
	 * Gets the root element local name.
	 *
	 * @return Root element local name.
	 */
	@Override
	public String getRootName() {
		return this.elmRootLocalName;
	}

	/**
	 * Sets the root element XML local name.
	 * 
	 * @param aRootName
	 *           Root element XML local name.
	 */
	public void setRootName(String aRootName) {
		this.elmRootLocalName = aRootName;
	}

	/**
	 * Gets the minimum element XML local name.
	 *
	 * @return Minimum element XML local name.
	 */
	@Override
	public String getElmMin() {
		return this.elmMinLocalName;
	}

	/**
	 * Sets the minimum element XML local name.
	 * 
	 * @param aElmMin
	 *           The minimum element XML local name.
	 */
	public void setElmMin(String aElmMin) {
		this.elmMinLocalName = aElmMin;
	}

	/**
	 * Gets the maximum element local name.
	 *
	 * @return Maximum element local name.
	 */
	@Override
	public String getElmMax() {
		return this.elmMaxLocalName;
	}

	/**
	 * Sets the maximum element XML local name.
	 * 
	 * @param aElmMin
	 *           The maximum element XML local name.
	 */
	public void setElmMax(String aElmMax) {
		this.elmMaxLocalName = aElmMax;
	}

	/**
	 * Gets the {@link Logger} to use.
	 *
	 * @return {@link Logger} instance to use for logging.
	 */
	@Override
	protected Logger getLogger() {
		return cat;
	}

	/**
	 * Validate the state of the {@code EPPRegistryMinMaxLength} instance. A
	 * valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the {@code EPPCodecException} will contain a
	 * description of the error.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	protected void validateState() throws EPPEncodeException {
		// Root Local Name
		if (this.elmRootLocalName == null) {
			throw new EPPEncodeException("Root local name not set");
		}
	}

}
