/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.util.EPPCatFactory;

/**
 * The default period length if not defined by the client.
 */
public class EPPRegistryDefaultPeriodType extends EPPRegistryPeriodType {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryDefaultPeriodType.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryDefaultPeriodType}.
	 */
	public static final String ELM_LOCALNAME = "default";

	/**
	 * XML root tag for {@code EPPRegistryDefaultPeriodType}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	public EPPRegistryDefaultPeriodType() {
	}

	/**
	 * Constructs {@code EPPRegistryDefaultPeriodType} with a number and unit
	 * using an {@code Integer}.
	 *
	 * @param aNumber
	 *           Default period number
	 * @param aUnit
	 *           Default period unit
	 */
	public EPPRegistryDefaultPeriodType(Integer aNumber, Unit aUnit) {
		super.number = aNumber;
		super.unit = aUnit;
	}

	/**
	 * Constructs {@code EPPRegistryDefaultPeriodType} with a number and unit
	 * using an {@code int}.
	 *
	 * @param aNumber
	 *           Default period number
	 * @param aUnit
	 *           Default period unit
	 */
	public EPPRegistryDefaultPeriodType(int aNumber, Unit aUnit) {
		this();
		super.number = Integer.valueOf(aNumber);
		super.unit = aUnit;
	}

	/**
	 * Extra validation of the default period number and unit.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	@Override
	void extraValidate() throws EPPCodecException {
		// Number
		int theNumber = this.number.intValue();
		if (theNumber < 1 || theNumber > 99) {
			throw new EPPCodecException(getRootName() + ": number should be between 1 - 99.");
		}

		// Unit
		if (super.unit != Unit.y && super.unit != Unit.m) {
			throw new EPPCodecException(getRootName() + ": invalid unit. Valid values: y/m");
		}
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}
	
	/**
	 * Gets the root element local name.
	 * 
	 * @return Root element local name.
	 */
	protected String getRootName() {
		return ELM_LOCALNAME;
	}	

}
