/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Registry &lt;registry:chkData&gt; response to a
 * {@code EPPRegistryCheckCmd}. When a &lt;check&gt; command has been processed
 * successfully, the EPP &lt;resData&gt; element MUST contain a child
 * &lt;registry:chkData&gt; element that identifies the registry namespace and
 * the location of the registry schema. The &lt;registry:chkData&gt; elements
 * that contain the following child elements: <br>
 *
 * <ul>
 * <li>A &lt;registry:name&gt; element that contains the fully qualified name of
 * the queried zone object. This element MUST contain an "avail" attribute whose
 * value indicates object availability at the moment the &lt;check&gt; command
 * was completed. A value of "1" or "true" means that the object is availabe. A
 * value of "0" or "false" means that the object is not available.</li>
 * <li>An OPTIONAL &lt;registry:reason&gt; element that MAY be provided when an
 * object is not available for provisioning. If present, this element contains
 * server-specific text to help explain why the object is unavailable. This text
 * MUST be represented in the response language previously negotiated with the
 * client; an OPTIONAL "lang" attribute MAY be present to identify the language
 * if the negotiated value is something other that a default value of "en"
 * (English).</li>
 * </ul>
 *
 * <br>
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryCheckCmd
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryCheckResult
 */
public class EPPRegistryCheckResp extends EPPResponse {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryCheckResp.class);

	/**
	 * XML local name for {@code EPPRegistryCheckResp}.
	 */
	public static final String ELM_LOCALNAME = "chkData";

	/**
	 * XML root tag for {@code EPPRegistryCheckResp}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * {@code List} of {@link EPPRegistryCheckResult} instances.
	 */
	private List<EPPRegistryCheckResult> checkResults = new ArrayList<EPPRegistryCheckResult>();

	/**
	 * {@code EPPRegistryCheckResp} default constructor. It will set results
	 * attribute to an empty {@code List}.
	 */
	public EPPRegistryCheckResp() {
	}

	/**
	 * {@code EPPRegistryCheckResp} constructor that will set the result of an
	 * individual zone object.
	 *
	 * @param aTransId
	 *           transaction Id associated with response
	 * @param aResult
	 *           {@link EPPRegistryCheckResult} instance of a single registry
	 *           name
	 */
	public EPPRegistryCheckResp(EPPTransId aTransId, EPPRegistryCheckResult aResult) {
		super(aTransId);

		this.checkResults.add(aResult);
	}

	/**
	 * {@code EPPRegistryCheckResp} constructor that will set the result of
	 * multiple zone objects.
	 *
	 * @param aTransId
	 *           transaction Id associated with response
	 * @param aResults
	 *           {@code List} of {@link EPPRegistryCheckResult} instances
	 */
	public EPPRegistryCheckResp(EPPTransId aTransId, List aResults) {
		super(aTransId);

		this.setCheckResults(aResults);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryCheckResp} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryCheckResp} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryCheckResp} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		if (this.checkResults == null || this.checkResults.size() == 0) {
			throw new EPPEncodeException("Invalid state on EPPRegistryInfoResp.encode: chekResults is empty");
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Results
		EPPUtil.encodeCompList(aDocument, root, this.checkResults);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryCheckResp} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryCheckResp} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Results
		this.checkResults = EPPUtil.decodeCompList(aElement, EPPRegistryMapFactory.NS, EPPRegistryCheckResult.ELM_NAME,
		      EPPRegistryCheckResult.class);
	}

	/**
	 * Get the EPP response type associated with {@code EPPRegistryCheckResp}.
	 *
	 * @return {@code EPPRegistryCheckResp.ELM_NAME}
	 */
	@Override
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Get the EPP command Namespace associated with {@code EPPRegistryCheckResp}
	 * .
	 *
	 * @return {@code EPPRegistryMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Compare an instance of {@code EPPRegistryCheckResp} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryCheckResp)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPRegistryCheckResp theCheckData = (EPPRegistryCheckResp) aObject;

		// results
		if (!EPPUtil.equalLists(this.checkResults, theCheckData.checkResults)) {
			cat.error("EPPRegistryCheckResp: checkResults not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryCheckResp}.
	 *
	 * @return clone of {@code EPPRegistryCheckResp}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryCheckResp clone = (EPPRegistryCheckResp) super.clone();

		clone.checkResults = (List) ((ArrayList) this.checkResults).clone();

		for (int i = 0; i < this.checkResults.size(); i++) {
			if (this.checkResults.get(i) == null) {
				clone.checkResults.set(i, null);
			}
			else {
				clone.checkResults.set(i,
				      (EPPRegistryCheckResult) ((EPPRegistryCheckResult) (this.checkResults.get(i))).clone());
			}
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Get the check results of zone objects.
	 *
	 * @return {@code List} of {@link EPPRegistryCheckResult}
	 */
	public List<EPPRegistryCheckResult> getCheckResults() {
		return this.checkResults;
	}

	/**
	 * Set the check results of zone objects.
	 *
	 * @param aResults
	 *           {@code List} of {@link EPPRegistryCheckResult}
	 */
	public void setCheckResults(List<EPPRegistryCheckResult> aResults) {
		if (aResults == null) {
			this.checkResults = new ArrayList<EPPRegistryCheckResult>();
		}
		else {
			this.checkResults = aResults;
		}
	}
}
