/***********************************************************
Copyright (C) 2019 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.interfaces.v04;

import java.util.Random;

import org.apache.log4j.Logger;

import com.verisign.epp.codec.gen.EPPCodecTst;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.loginsec.v03.EPPLoginSecData;
import com.verisign.epp.interfaces.EPPApplicationSingle;
import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPLoginSecLoginAdapter;
import com.verisign.epp.interfaces.EPPSession;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.TestThread;

import junit.framework.Assert;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Is a unit test the use of the {@link EPPLoginSecLoginAdapter} with the
 * {@link EPPSession} and exercising the login security handler for returning
 * specific security events based on the login identifier (clientId).<br>
 * The unit test can be run in multiple threads by setting the "threads" system
 * property. For example, the unit test can be run in 2 threads with the
 * configuration file ../../epp.config with the following command: <br>
 * <br>
 * java com.verisign.epp.interfaces.EPPLoginSecTst -Dthreads=2 ../../epp.config
 * <br>
 * <br>
 * The unit test is dependent on the use of <a
 * href=http://www.mcwestcorp.com/Junit.html>JUNIT 3.5</a><br>
 */
public class EPPLoginSecTst extends TestCase {
	/**
	 * Handle to the Singleton EPP Application instance (
	 * <code>EPPApplicationSingle</code>)
	 */
	private static EPPApplicationSingle app = EPPApplicationSingle.getInstance();

	/** Name of configuration file to use for test (default = epp.config). */
	private static String configFileName = "epp.config";

	/** Logging category */
	private static final Logger cat = Logger.getLogger(EPPLoginSecTst.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	private EPPSession session = null;

	/** Current test iteration */
	private int iteration = 0;

	/**
	 * Random instance for the generation of unique objects (hosts, IP addresses,
	 * etc.).
	 */
	private Random rd = new Random(System.currentTimeMillis());

	/**
	 * Allocates an <code>EPPFeeDomainTst</code> with a logical name. The
	 * constructor will initialize the base class <code>TestCase</code> with the
	 * logical name.
	 *
	 * @param name
	 *           Logical name of the test
	 */
	public EPPLoginSecTst(String name) {
		super(name);
	}

	/**
	 * JUNIT <code>setUp</code> method, which sets the default client Id to
	 * "theRegistrar".
	 */
	protected void setUp() {
	}

	/**
	 * JUNIT <code>tearDown</code>, which currently does nothing.
	 */
	protected void tearDown() {
	}

	/**
	 * Test passing a short password during a login that should not set the
	 * password in the login security extension, but will include the 
	 * user agent as long as it has been enabled.
	 */
	public void testStandardLogin() {
		printStart("testStandardLogin");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("StandardLogin");
			this.session.setPassword("foo-BAR2");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		this.endSession();

		printEnd("testStandardLogin");
	}

	/**
	 * Unit test for the extension to the login command with long current
	 * password.
	 */
	public void testLoginCmdLongPwd() {
		EPPCodecTst.printStart("testLoginCmdLongPwd");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("LongPwd");
			this.session.setPassword("this is a long password");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		this.endSession();

		EPPCodecTst.printEnd("testLoginCmdLongPwd");
	}

	/**
	 * Unit test for the extension to the login command with long new password.
	 */
	public void testLoginCmdLongNewPwd() {
		EPPCodecTst.printStart("testLoginCmdLongNewPwd");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("LongNewPwd");
			this.session.setPassword("foo-BAR2");
			this.session.setNewPassword("new password that is long");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		this.endSession();

		EPPCodecTst.printEnd("testLoginCmdLongNewPwd");
	}

	/**
	 * Unit test for the extension to the login command with long current and new
	 * password.
	 */
	public void testLoginCmdLongPwdNewPwd() {
		EPPCodecTst.printStart("testLoginCmdLongPwdNewPwd");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("LongPwdNewPwd");
			this.session.setPassword("this is a long password");
			this.session.setNewPassword("new password that is long");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		this.endSession();

		EPPCodecTst.printEnd("testLoginCmdLongPwdNewPwd");
	}

	/**
	 * Unit test for the extension to the login response by sending the login
	 * command with the login identifier (clientId) set to "test-expiring-pw".
	 * The response must have a {@link EPPLoginSecData} extension set by the
	 * server.
	 */
	public void testExpiringPasswordResponse() {
		EPPCodecTst.printStart("testExpiringPasswordResponse");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("test-expiring-pw");
			this.session.setPassword("this is a long password");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();

			EPPResponse theResponse = this.session.getResponse();

			if (theResponse.hasExtension(EPPLoginSecData.class)) {
				printMsg("testExpiringPasswordResponse v2 login security response extension = "
				      + theResponse.getExtension(EPPLoginSecData.class));
			}

		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		this.endSession();

		EPPCodecTst.printEnd("testExpiringPasswordResponse");
	}

	/**
	 * Unit test for the extension to the login response by sending the login
	 * command with the login identifier (clientId) set to "test-expired-pw". The
	 * login must fail and the response must have a {@link EPPLoginSecData}
	 * extension set by the server in the failed response.
	 */
	public void testExpiredPasswordResponse() {
		EPPCodecTst.printStart("testExpiredPasswordResponse");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("test-expired-pw");
			this.session.setPassword("this is a long password");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();

			Assert.fail("Login with test-expired-pw should fail");
		}
		catch (EPPCommandException e) {

			EPPResponse theResponse = this.session.getResponse();

			if (theResponse.hasExtension(EPPLoginSecData.class)) {
				printMsg("testExpiredPasswordResponse v2 login security response extension = "
				      + theResponse.getExtension(EPPLoginSecData.class));
			}
		}

		EPPCodecTst.printEnd("testExpiredPasswordResponse");
	}

	/**
	 * Unit test for the extension to the login response by sending the login
	 * command with the login identifier (clientId) set to "test-all-events". The
	 * response must have a {@link EPPLoginSecData} extension set by the server.
	 */
	public void testWarningAllResponse() {
		EPPCodecTst.printStart("testWarningAllResponse");

		try {
			this.session = new EPPSession();

			// Set attributes for initSession
			this.session.setClientID("test-all-events");
			this.session.setPassword("this is a long password");
			this.session.setTransId("ABC-12345-XYZ");
			this.session.setVersion("1.0");
			this.session.setLang("en");

			this.session.initSession();

			EPPResponse theResponse = this.session.getResponse();

			if (theResponse.hasExtension(EPPLoginSecData.class)) {
				printMsg("testWarningAllResponse v2 login security response extension = "
				      + theResponse.getExtension(EPPLoginSecData.class));
			}

		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		this.endSession();

		EPPCodecTst.printEnd("testWarningAllResponse");
	}

	/**
	 * JUNIT <code>suite</code> static method, which returns the tests associated
	 * with <code>EPPFeeDomainTst</code>.
	 *
	 * @return Test suite
	 */
	public static Test suite() {
		TestSuite suite = new TestSuite(EPPLoginSecTst.class);

		String theConfigFileName = System.getProperty("EPP.ConfigFile");
		if (theConfigFileName != null)
			configFileName = theConfigFileName;

		try {
			app.initialize(configFileName);
		}

		catch (EPPCommandException e) {
			e.printStackTrace();

			Assert.fail("Error initializing the EPP Application: " + e);
		}

		return suite;
	}

	/**
	 * Handle an <code>EPPCommandException</code>, which can be either a server
	 * generated error or a general exception. If the exception was caused by a
	 * server error, "Server Error :&lt;Response XML&gt;" will be specified. If
	 * the exception was caused by a general algorithm error, "General Error
	 * :&lt;Exception Description&gt;" will be specified.
	 *
	 * @param aException
	 *           Exception thrown during test
	 */
	public void handleException(Exception aException) {
		EPPResponse response = session.getResponse();

		aException.printStackTrace();

		// Is a server specified error?
		if ((response != null) && (!response.isSuccess())) {
			Assert.fail("Server Error : " + response);
		}

		else {
			Assert.fail("General Error : " + aException);
		}
	}

	/**
	 * Logout of active session.
	 */
	private void endSession() {
		printStart("endSession");

		this.session.setTransId("ABC-12345-XYZ");

		// End the session
		try {
			this.session.endSession();
		}

		catch (EPPCommandException e) {
			EPPResponse response = this.session.getResponse();

			// Is a server specified error?
			if ((response != null) && (!response.isSuccess())) {
				Assert.fail("Server Error : " + response);
			}

			else // Other error
			{
				e.printStackTrace();

				Assert.fail("initSession Error : " + e);
			}
		}

		printEnd("endSession");
	}

	/**
	 * Unit test main, which accepts the following system property options: <br>
	 * 
	 * <ul>
	 * <li>iterations Number of unit test iterations to run</li>
	 * <li>validate Turn XML validation on (<code>true</code>) or off (
	 * <code>false</code>). If validate is not specified, validation will be off.
	 * </li>
	 * </ul>
	 * 
	 *
	 * @param args
	 *           Command line arguments
	 */
	public static void main(String[] args) {
		// Override the default configuration file name?
		if (args.length > 0) {
			configFileName = args[0];
		}

		// Number of Threads
		int numThreads = 1;

		String threadsStr = System.getProperty("threads");

		if (threadsStr != null) {
			numThreads = Integer.parseInt(threadsStr);

			// Run test suite in multiple threads?
		}

		if (numThreads > 1) {
			// Spawn each thread passing in the Test Suite
			for (int i = 0; i < numThreads; i++) {
				TestThread thread = new TestThread("EPPSessionTst Thread " + i, EPPLoginSecTst.suite());

				thread.start();
			}
		}

		else { // Single threaded mode.
			junit.textui.TestRunner.run(EPPLoginSecTst.suite());
		}

		try {
			app.endApplication();
		}

		catch (EPPCommandException e) {
			e.printStackTrace();

			Assert.fail("Error ending the EPP Application: " + e);
		}
	}

	/**
	 * Print the start of a test with the <code>Thread</code> name if the current
	 * thread is a <code>TestThread</code>.
	 *
	 * @param aTest
	 *           name for the test
	 */
	private void printStart(String aTest) {
		if (Thread.currentThread() instanceof TestThread) {
			System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

			cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " Start");
		}

		System.out.println("Start of " + aTest);

		System.out.println("****************************************************************\n");
	}

	/**
	 * Print the end of a test with the <code>Thread</code> name if the current
	 * thread is a <code>TestThread</code>.
	 *
	 * @param aTest
	 *           name for the test
	 */
	private void printEnd(String aTest) {
		System.out.println("****************************************************************");

		if (Thread.currentThread() instanceof TestThread) {
			System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

			cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " End");
		}

		System.out.println("End of " + aTest);

		System.out.println("\n");
	}

	/**
	 * Print message
	 *
	 * @param aMsg
	 *           message to print
	 */
	private void printMsg(String aMsg) {
		if (Thread.currentThread() instanceof TestThread) {
			System.out.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

			cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
		}

		else {
			System.out.println(aMsg);

			cat.info(aMsg);
		}
	}

	/**
	 * Print error message
	 *
	 * @param aMsg
	 *           errpr message to print
	 */
	private void printError(String aMsg) {
		if (Thread.currentThread() instanceof TestThread) {
			System.err.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

			cat.error(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
		}

		else {
			System.err.println(aMsg);

			cat.error(aMsg);
		}
	}
}
