/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/
package com.verisign.epp.codec.loginsec.v01;

import java.security.InvalidParameterException;

/**
 * Event types, which include the following possible values:<br>
 * <ul>
 * <li><code>PASSWORD</code> - Password expiry event type.</li>
 * <li><code>CERTIFICATE</code> - Client certificate expiry event type.</li>
 * <li><code>CIPHER</code> - Insecure or deprecated TLS cipher suite event type.
 * </li>
 * <li><code>TLS_PROTOCOL</code> - Insecure or deprecated TLS protocol event
 * type.</li>
 * <li><code>NEW_PW</code> - New password does not meet server password
 * complexity requirements event type.</li>
 * <li><code>STAT</code> - Login security statistical warning event type.</li>
 * <li><code>CUSTOM</code> - Custom event type attribute like the
 * &quot;customName&quot; attribute.</li>
 * </ul>
 */
public enum EventType {
	PASSWORD("password"), CERTIFICATE("certificate"), CIPHER("cipher"), TLS_PROTOCOL("tlsProtocol"), NEW_PW(
	      "newPw"), STAT("stat"), CUSTOM("custom");

	private final String typeStr;

	/**
	 * Define the string value for the enumerated value.
	 * 
	 * @param aTypeStr
	 *           Enumerated value string
	 */
	EventType(String aTypeStr) {
		this.typeStr = aTypeStr;
	}

	/**
	 * Get the event type enumerated value given the matching string.
	 * 
	 * @param aString
	 *           <code>EventType</code> enumerated string to convert to an
	 *           enumerated <code>EventType</code> instance.
	 * 
	 * @return Enumerated <code>EventType</code> value matching the
	 *         <code>String</code>.
	 * 
	 * @throws InvalidParameterException
	 *            If <code>aString</code> does not match an enumerated
	 *            <code>EventType</code> string value.
	 */
	public static EventType getEventType(String aString) {
		if (aString.equals(PASSWORD.typeStr)) {
			return PASSWORD;
		}
		else if (aString.equals(CERTIFICATE.typeStr)) {
			return CERTIFICATE;
		}
		else if (aString.equals(CIPHER.typeStr)) {
			return CIPHER;
		}
		else if (aString.equals(TLS_PROTOCOL.typeStr)) {
			return TLS_PROTOCOL;
		}
		else if (aString.equals(NEW_PW.typeStr)) {
			return NEW_PW;
		}
		else if (aString.equals(STAT.typeStr)) {
			return STAT;
		}
		else if (aString.equals(CUSTOM.typeStr)) {
			return CUSTOM;
		}
		else {
			throw new InvalidParameterException("Event type enum value of " + aString + " is not valid.");
		}
	}

	/**
	 * Convert the enumerated <code>EventType</code> value to a
	 * <code>String</code>.
	 */
	public String toString() {
		return this.typeStr;
	}

}
