/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.launchpolicy.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.codec.launchpolicy.v01.EPPLaunchPolicyPhase.Phase;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPLaunchPolicyInfoPhase} defines a supported &lt;launch:phase&gt;
 * value that can be passed in the phase.
 */
public class EPPLaunchPolicyInfoPhase implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPLaunchPolicyInfoPhase.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPLaunchPolicyInfoPhase}.
	 */
	public static final String ELM_LOCALNAME = "infoPhase";

	/**
	 * XML root tag for {@code EPPLaunchPolicyInfoPhase}.
	 */
	public static final String ELM_NAME = EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Phase type attribute name
	 */
	private static final String ATTR_TYPE = "type";

	/**
	 * OPTIONAL attribute name that can define a sub-phase or the full name of
	 * the phase type when the phase type has the "custom" value.
	 */
	private static final String ATTR_NAME = "name";

	/**
	 * Launch phase name using one of the {@link EPPLaunchPolicyPhase.Phase}
	 * constants.
	 */
	private EPPLaunchPolicyPhase.Phase phase;

	/**
	 * OPTIONAL "name" attribute that can define a sub-phase or the full name of
	 * the phase when the phase has the {@link EPPLaunchPolicyPhase.Phase#custom}
	 * value.
	 */
	private String name;

	/**
	 * Default constructor for {@code EPPLaunchPolicyInfoPhase}.
	 */
	public EPPLaunchPolicyInfoPhase() {
	}

	/**
	 * Constructor for {@code EPPLaunchPolicyInfoPhase} that takes the required
	 * phase attribute.
	 *
	 * @param aPhase
	 *           The launch phase
	 *
	 */
	public EPPLaunchPolicyInfoPhase(Phase aPhase) {
		this.phase = aPhase;
	}

	/**
	 * Constructor for {@code EPPLaunchPolicyInfoPhase} that takes the required
	 * phase attribute and the optional name attribute.
	 *
	 * @param aPhase
	 *           The launch phase
	 *
	 * @param aName
	 *           The sub-phase name or the custom name if {@code aPhase} is
	 *           {@link EPPLaunchPolicyPhase.Phase#custom}. Set to {@code null}
	 *           if undefined.
	 */
	public EPPLaunchPolicyInfoPhase(Phase aPhase, String aName) {
		this.phase = aPhase;
		this.name = aName;
	}

	/**
	 * encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding {@code EPPLaunchPolicyInfoPhase}
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (this.phase == null) {
			throw new EPPEncodeException("Undefined phase in EPPLaunchPolicyInfoPhase");
		}

		Element root = aDocument.createElementNS(EPPLaunchPolicyExtFactory.NS, ELM_NAME);

		// Phase
		root.setAttribute(ATTR_TYPE, this.phase.toString());

		// Name
		if (this.hasName()) {
			root.setAttribute(ATTR_NAME, this.name);
		}

		return root;
	}

	/**
	 * decode a DOM element tree to initialize the instance attributes. The
	 * {@code aElement} argument represents the root DOM element and is used to
	 * traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           {@code Element} to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding {@code Element}
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Phase
		this.phase = Phase.valueOf(EPPUtil.decodeStringAttr(aElement, ATTR_TYPE));

		// Name
		this.name = EPPUtil.decodeStringAttr(aElement, ATTR_NAME);
	}

	/**
	 * clone an {@code EPPCodecComponent}.
	 *
	 * @return clone of concrete {@code EPPOrgExt}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPLaunchPolicyInfoPhase clone = (EPPLaunchPolicyInfoPhase) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of {@code EPPOrgExt} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLaunchPolicyInfoPhase)) {
			return false;
		}

		EPPLaunchPolicyInfoPhase other = (EPPLaunchPolicyInfoPhase) aObject;

		// Phase
		if (!EqualityUtil.equals(this.phase, other.phase)) {
			cat.error("EPPLaunchPolicyInfoPhase.equals(): phase not equal");
			return false;
		}

		// Name
		if (!EqualityUtil.equals(this.name, other.name)) {
			cat.error("EPPLaunchPolicyInfoPhase.equals(): name not equal");
			return false;
		}

		return true;
	}

	/**
	 * Is the phase defined?
	 *
	 * @return {@code true} if the phase is defined; {@code false} otherwise.
	 */
	public boolean hasPhase() {
		return (this.phase != null ? true : false);
	}

	/**
	 * Gets the phase value.
	 *
	 * @return {@link EPPLaunchPolicyPhase.Phase} enumerated value if set;
	 *         {@code null} otherwise.
	 */
	public EPPLaunchPolicyPhase.Phase getPhase() {
		return this.phase;
	}

	/**
	 * Sets the phase value.
	 *
	 * @param aPhase
	 *           Sets the phase value.
	 */
	public void setPhase(EPPLaunchPolicyPhase.Phase aPhase) {
		this.phase = aPhase;
	}

	/**
	 * Is the name defined?
	 *
	 * @return {@code true} if the name is defined; {@code false} otherwise.
	 */
	public boolean hasName() {
		return (this.name != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL phase name or sub-phase name.
	 *
	 * @return The phase name or sub-phase name if defined; {@code null}
	 *         otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the OPTIONAL phase name or sub-phase name.
	 *
	 * @param aName
	 *           Phase name
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPLaunchPolicyExtFactory.NS;
	}

}
