/***********************************************************
Copyright (C) 2017 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/
package com.verisign.epp.codec.fee.v23;

import java.security.InvalidParameterException;


/**
 * Phases, which include the following possible values:<br>
 * <ul>
 * <li><code>SUNRISE</code> - Phase when trademark holders can submit registrations or applications with
 * trademark information that can be validated by.</li>
 * <li><code>LANDRUSH</code> - Post sunrise phase when non-trademark holders are allowed to register
 * domain names with steps taken to address a large volume of initial
 * registrations.</li>
 * <li><code>CLAIMS</code> - Trademark claims as defined by Trademark Clearinghouse model of displaying
 * a claims notice to clients for domain names that match trademarks.</li>
 * <li><code>OPEN</code> - Post launch phase that is also referred to as "steady state". Servers MAY
 * require additional trademark protection with this phase.</li>
 * <li><code>CUSTOM</code> - Custom phase that is defined using a free-form
 * attribute like the &quot;subphase&quot; attribute.</li>
 * </ul>
 */
public enum Phase {
	SUNRISE("sunrise"), LANDRUSH("landrush"), CLAIMS("claims"), OPEN("open"), CUSTOM("custom");

	private final String phaseStr;

	/**
	 * Define the string value for the enumerated value.
	 * 
	 * @param aPhaseStr
	 *           Enumerated value string
	 */
	Phase(String aPhaseStr) {
		this.phaseStr = aPhaseStr;
	}

	/**
	 * Get the phase enumerated value given the matching string.
	 * 
	 * @param aString
	 *           <code>Phase</code> enumerated string to convert to an
	 *           enumerated <code>Phase</code> instance.
	 * 
	 * @return Enumerated <code>Phase</code> value matching the
	 *         <code>String</code>.
	 * 
	 * @throws InvalidParameterException
	 *            If <code>aString</code> does not match an enumerated
	 *            <code>Phase</code> string value.
	 */
	public static Phase getPhase(String aString) {
		if (aString.equals(SUNRISE.phaseStr)) {
			return SUNRISE;
		}
		else if (aString.equals(LANDRUSH.phaseStr)) {
			return LANDRUSH;
		}
		else if (aString.equals(CLAIMS.phaseStr)) {
			return CLAIMS;
		}
		else if (aString.equals(OPEN.phaseStr)) {
			return OPEN;
		}
		else if (aString.equals(CUSTOM.phaseStr)) {
			return CUSTOM;
		}
		else {
			throw new InvalidParameterException("Phase enum value of " + aString + " is not valid.");
		}
	}

	/**
	 * Convert the enumerated <code>Phase</code> value to a
	 * <code>String</code>.
	 */
	public String toString() {
		return this.phaseStr;
	}

}
