/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.validate.v02;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPValidateCheckResult} represents the result of an individual contact
 * validation. The attributes of {@code EPPValidateCheckResult} include the
 * contact identifier, the result code, and zero or more key value pairs.
 *
 * @see com.verisign.epp.codec.validate.v02.EPPValidateCheckResp
 */
public class EPPValidateCheckResult implements EPPCodecComponent {
	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPValidateCheckResult.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Constant for the result local name
	 */
	public static final String ELM_LOCALNAME = "cd";

	/**
	 * Constant for the result tag
	 */
	public static final String ELM_NAME = EPPValidateMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the contact identifier element
	 */
	private static final String ELM_ID = "id";

	/**
	 * XML local name for the result code
	 */
	private static final String ELM_CODE = "response";

	/**
	 * The contact identifier.
	 */
	private String id;

	/**
	 * Result code must use one of the
	 * {@link com.verisign.epp.codec.gen.EPPResult} result code constant values.
	 */
	private int code = -1;

	/**
	 * Optional list of key value pairs used to define specific validation
	 * errors.
	 */
	private List<EPPValidateKeyValue> keyValues = new ArrayList<EPPValidateKeyValue>();

	/**
	 * Default constructor for {@code EPPValidateCheckResult}.
	 */
	public EPPValidateCheckResult() {
	}

	/**
	 * Constructor for {@code EPPValidateCheckResult} that includes the required
	 * attributes of the contact identifier and the result code.
	 *
	 * @param aId
	 *           Contact identifier
	 * @param aResultCode
	 *           Must be one of the {@link com.verisign.epp.codec.gen.EPPResult}
	 *           result code constant values.
	 */
	public EPPValidateCheckResult(String aId, int aCode) {
		this.id = aId;
		this.code = aCode;
	}

	/**
	 * Gets the contact identifier.
	 *
	 * @return The contact identifier if defined;{@code null} otherwise.
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * Sets the contact identifier.
	 *
	 * @param aId
	 *           The contact identifier
	 */
	public void setId(String aId) {
		this.id = aId;
	}

	/**
	 * Gets the result code.
	 *
	 * @return Must be one of the {@link com.verisign.epp.codec.gen.EPPResult}
	 *         result code constant values if defined; {@code -1} otherwise.
	 */
	public int getCode() {
		return this.code;
	}

	/**
	 * Sets the result code.
	 *
	 * @param aCode
	 *           Must be one of the {@link com.verisign.epp.codec.gen.EPPResult}
	 *           result code constant values.
	 */
	public void setCode(int aCode) {
		this.code = aCode;
	}

	/**
	 * Adds a key value pair to the list of key value pairs. The key value pair
	 * is used to defined specific validation errors.
	 * 
	 * @param aKeyValue
	 *           Key value pair to add to the list
	 */
	public void addKeyValue(EPPValidateKeyValue aKeyValue) {
		this.keyValues.add(aKeyValue);
	}

	/**
	 * Sets the full list of key value pairs. A key value pair is used to define
	 * specific validation errors.
	 *
	 * @param aKeyValues
	 *           List of {@code EPPValidateKeyValue} instances.
	 */
	public void setKeyValues(List<EPPValidateKeyValue> aKeyValues) {
		if (aKeyValues == null) {
			this.keyValues = new ArrayList<EPPValidateKeyValue>();
		}
		else {
			this.keyValues = aKeyValues;
		}
	}

	/**
	 * Gets the full list of key value pairs. A key value pair is used to define
	 * specific validation errors.
	 *
	 *
	 * @return List of {@code EPPValidateKeyValue} instances.
	 */
	public List<EPPValidateKeyValue> getKeyValues() {
		return this.keyValues;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPValidateCheckResult} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPValidateCheckResult} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPValidateCheckResult} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Check required attributes
		if (this.id == null) {
			throw new EPPEncodeException("Undefined id in EPPValidateCheckResult");
		}
		if (this.code == -1) {
			throw new EPPEncodeException("Undefined code in EPPValidateCheckResult");
		}

		Element root = aDocument.createElementNS(EPPValidateMapFactory.NS, ELM_NAME);

		// Contact Identifier
		EPPUtil.encodeString(aDocument, root, this.id, EPPValidateMapFactory.NS,
		      EPPValidateMapFactory.NS_PREFIX + ":" + ELM_ID);

		// Code
		EPPUtil.encodeInteger(aDocument, root, this.code, EPPValidateMapFactory.NS,
		      EPPValidateMapFactory.NS_PREFIX + ":" + ELM_CODE);

		// Key Values
		EPPUtil.encodeCompList(aDocument, root, this.keyValues);

		return root;
	}

	/**
	 * Decode the {@code EPPValidateCheckResult} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPValidateCheckResult} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Contact Identifier
		this.id = EPPUtil.decodeString(aElement, EPPValidateMapFactory.NS, ELM_ID);

		// Code
		this.code = EPPUtil.decodeInteger(aElement, EPPValidateMapFactory.NS, ELM_CODE);

		// Key Values
		this.keyValues = EPPUtil.decodeCompList(aElement,
				EPPValidateMapFactory.NS,
				EPPValidateKeyValue.ELM_NAME,
				EPPValidateKeyValue.class);
	}

	/**
	 * Compare an instance of {@code EPPValidateCheckResult} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPValidateCheckResult)) {
			return false;
		}

		EPPValidateCheckResult other = (EPPValidateCheckResult) aObject;

		// Contact Identifier
		if (!EqualityUtil.equals(this.id, other.id)) {
			cat.error("EPPValidateCheckResult.equals(): id not equal");
			return false;
		}

		// Code
		if (!EqualityUtil.equals(this.code, other.code)) {
			cat.error("EPPValidateCheckResult.equals(): code not equal");
			return false;
		}

		// Key Values
		if (!EqualityUtil.equals(this.keyValues, other.keyValues)) {
			cat.error("EPPValidateCheckResult.equals(): keyValues not equal");
			return false;
		}
		
		return true;
	}

	/**
	 * Clone {@code EPPValidateCheckResult}.
	 *
	 * @return clone of {@code EPPValidateCheckResult}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPValidateCheckResult clone = null;

		clone = (EPPValidateCheckResult) super.clone();

		// Key Values
		clone.keyValues = new ArrayList<EPPValidateKeyValue>(
				this.keyValues.size());
		for (EPPValidateKeyValue item : this.keyValues)
			clone.keyValues.add((EPPValidateKeyValue) item.clone());
		
		
		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPValidateMapFactory.NS;
	}

}
