/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.validate.v02;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCheckCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * The {@code EPPValidateCheckCmd} is used for the Validate Check Command, that
 * is used for the validation of contact and eligibility data.
 *
 * @see com.verisign.epp.codec.validate.v02.EPPValidateCheckResp
 */
public class EPPValidateCheckCmd extends EPPCheckCmd {
	/**
	 * Constant for the name verification check command
	 */
	public static final String ELM_LOCALNAME = "check";

	/**
	 * XML Element Name of {@code EPPValidateCheckCmd} root element.
	 */
	final static String ELM_NAME = EPPValidateMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPValidateCheckCmd.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Labels to check.
	 */
	private List<EPPValidateContact> contacts = new ArrayList<EPPValidateContact>();

	/**
	 * {@code EPPValidateCheckCmd} default constructor.
	 */
	public EPPValidateCheckCmd() {
	}

	/**
	 * {@code EPPValidateCheckCmd} constructor with just the client transaction
	 * identifier.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 */
	public EPPValidateCheckCmd(String aTransId) {
		super(aTransId);
	}

	/**
	 * {@code EPPValidateCheckCmd} constructor that will check an individual
	 * contact.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aContact
	 *           Contact to validate
	 */
	public EPPValidateCheckCmd(String aTransId, EPPValidateContact aContact) {
		super(aTransId);
		this.contacts.add(aContact);
	}

	/**
	 * {@code EPPValidateCheckCmd} constructor that will check a list of
	 * contacts.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aContacts
	 *           {@code List} of labels to check
	 */
	public EPPValidateCheckCmd(String aTransId, List<EPPValidateContact> aContacts) {
		super(aTransId);
		this.contacts = aContacts;
	}

	/**
	 * Gets the EPP command Namespace associated with {@code EPPValidateCheckCmd}
	 * .
	 *
	 * @return {@code EPPValidateMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPValidateMapFactory.NS;
	}

	/**
	 * Compare an instance of {@code EPPValidateCheckCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPValidateCheckCmd)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPValidateCheckCmd other = (EPPValidateCheckCmd) aObject;

		// Contacts
		if (!EqualityUtil.equals(this.contacts, other.contacts)) {
			cat.error("EPPValidateCheckCmd.equals(): contacts not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPValidateCheckCmd}.
	 *
	 * @return Deep copy clone of {@code EPPValidateCheckCmd}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPValidateCheckCmd clone = (EPPValidateCheckCmd) super.clone();

		if (this.hasContacts()) {
			clone.contacts = (List) ((ArrayList) this.contacts).clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Is there any contacts set?
	 *
	 * @return {@code true} if there is at least one {@link EPPValidateContact}
	 *         set; {@code false} otherwise.
	 */
	public boolean hasContacts() {
		if (this.contacts != null && !this.contacts.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Sets individual contact to validate.
	 *
	 * @param aContact
	 *           Contact to validate
	 */
	public void setContact(EPPValidateContact aContact) {
		this.contacts = new ArrayList<EPPValidateContact>();
		this.contacts.add(aContact);
	}

	/**
	 * Gets contacts to validate.
	 *
	 * @return List of contacts.
	 */
	public List<EPPValidateContact> getContacts() {
		return this.contacts;
	}

	/**
	 * Sets contacts to validate.
	 *
	 * @param aContacts
	 *           Contacts to validate.
	 */
	public void setLabels(List<EPPValidateContact> aContacts) {
		if (aContacts == null) {
			this.contacts = new ArrayList<EPPValidateContact>();
		}
		else {
			this.contacts = aContacts;
		}
	}

	/**
	 * Add a contact to the list of contacts to validate.
	 *
	 * @param aContact
	 *           Contact to add
	 */
	public void addContact(EPPValidateContact aContact) {
		this.contacts.add(aContact);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPValidateCheckCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPValidateCheckCmd} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPValidateCheckCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		if (this.contacts.size() == 0) {
			throw new EPPEncodeException("No labels specified in EPPValidateCheckCmd");
		}

		Element root = aDocument.createElementNS(EPPValidateMapFactory.NS, ELM_NAME);
		root.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:" + EPPValidateMapFactory.NS_PREFIX_CONTACT,
		      EPPValidateMapFactory.NS_CONTACT);

		// Contacts
		EPPUtil.encodeCompList(aDocument, root, this.contacts);

		return root;
	}

	/**
	 * Decode the {@code EPPValidateCheckCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPValidateCheckCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Contacts
		this.contacts = EPPUtil.decodeCompList(aElement, EPPValidateMapFactory.NS, EPPValidateContact.ELM_NAME,
		      EPPValidateContact.class);
	}

}
