/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.registry.v02;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents a list of zone summary info. Upon receiving an &lt;info&gt;
 * command, with a &lt;registry:all&gt; element in it, the server puts a
 * &lt;registry:zoneList&gt; element in the response. The list contains the zone
 * summary attributes of supported zones. Use {@code setZoneList} and
 * {@code getZoneList} to set and get zone list; use {@code addZone} to add zone
 * to existing list; and use {@code setZone} to add one zone in the zone list (
 * {@code setZone} will empty the existing zone list).<br>
 * <br>
 *
 * Each element in the list contains the following info:
 *
 * <ul>
 * <li>&lt;registry:name&gt; - fully qualified name of the zone</li>
 * <li>&lt;registry:crDate&gt; - date of zone object creation</li>
 * <li>&lt;registry:upDate&gt; - optional date of last update</li>
 * </ul>
 * 
 * @see com.verisign.epp.codec.registry.v02.EPPRegistryZoneSummary
 */
public class EPPRegistryZoneList implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryZoneList.class);

	/**
	 * XML local name for {@code EPPRegistryZoneList}.
	 */
	public static final String ELM_LOCALNAME = "zoneList";

	/**
	 * XML root tag for {@code EPPRegistryZoneList}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * {@code List} of {@link EPPRegistryZoneSummary} instance, which contains
	 * zone summary info
	 */
	List<EPPRegistryZoneSummary> zoneList = new ArrayList<EPPRegistryZoneSummary>();

	/**
	 * Default constructor. {@code zoneList} is initialized as an empty
	 * {@code List}.
	 */
	public EPPRegistryZoneList() {
	}

	/**
	 * Constructor that takes a zone list.
	 *
	 * @param aZoneList
	 *           {@code List} of zone summary
	 */
	public EPPRegistryZoneList(List<EPPRegistryZoneSummary> aZoneList) {
		this.setZoneList(aZoneList);
	}

	/**
	 * Constructor that takes one {@link EPPRegistryZoneSummary} instance.
	 * {@code zoneList} is initialized to have one element.
	 *
	 * @param zoneSummary
	 *           summary attributes of one zone
	 */
	public EPPRegistryZoneList(EPPRegistryZoneSummary zoneSummary) {
		this.zoneList.add(zoneSummary);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the EPPRegistryZoneList
	 * instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the EPPRegistryZoneList
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode EPPRegistryZoneList instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		EPPUtil.encodeCompList(aDocument, root, this.zoneList);

		return root;
	}

	/**
	 * Decode the EPPRegistryZoneList attributes from the aElement DOM Element
	 * tree.
	 *
	 * @param aElement
	 *           - Root DOM Element to decode EPPRegistryZoneList from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		this.zoneList = EPPUtil.decodeCompList(aElement, EPPRegistryMapFactory.NS, EPPRegistryZoneSummary.ELM_NAME,
		      EPPRegistryZoneSummary.class);

		if (this.zoneList == null) {
			this.zoneList = new ArrayList();
		}
	}

	/**
	 * implements a deep {@code EPPRegistryZoneList} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryZoneList} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryZoneList)) {
			return false;
		}

		EPPRegistryZoneList theComp = (EPPRegistryZoneList) aObject;

		if (!((this.zoneList == null) ? (theComp.zoneList == null)
		      : EPPUtil.equalLists(this.zoneList, theComp.zoneList))) {
			cat.error("EPPRegistryZoneList.equals(): zoneList not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryZoneList}.
	 *
	 * @return clone of {@code EPPRegistryZoneList}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryZoneList clone = (EPPRegistryZoneList) super.clone();

		if (this.zoneList != null) {
			clone.zoneList = (List) ((ArrayList) this.zoneList).clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Get the zone summary list.
	 *
	 * @return {@code List} of {@code EPPRegistryZone}
	 */
	public List<EPPRegistryZoneSummary> getZoneList() {
		return this.zoneList;
	}

	/**
	 * Set the zone summary list.
	 *
	 * @param aZoneList
	 *           {@code List} of {@code EPPRegistryZone}
	 */
	public void setZoneList(List<EPPRegistryZoneSummary> aZoneList) {
		if (aZoneList == null) {
			this.zoneList = new ArrayList<EPPRegistryZoneSummary>();
		}
		else {
			this.zoneList = aZoneList;
		}
	}

	/**
	 * Append one zone to the existing zone list.
	 *
	 * @param zone
	 *           zone to add
	 */
	public void addZone(EPPRegistryZoneSummary zone) {
		this.zoneList.add(zone);
	}

	/**
	 * Clear the existing zone list and add add one zone to the list
	 *
	 * @param zone
	 *           zone to add
	 */
	public void setZone(EPPRegistryZoneSummary zone) {
		this.zoneList = new ArrayList<EPPRegistryZoneSummary>();
		this.zoneList.add(zone);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
