/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUpdateCmd;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents an EPP Registry &lt;update&gt; command. The EPP &lt;update&gt;
 * command provides a transform operation that allows a client to modify the
 * attributes of a zone object. In addition to the standard EPP command
 * elements, the &lt;update&gt; command MUST contain a &lt;domain:update&gt;
 * element that identifies the domain namespace and the location of the domain
 * schema. In addition to The &lt;registry:update&gt; element SHALL contain the
 * following child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;registry:zone&gt; element that contains the detailed registry
 * information of the object to be updated. Use {@code getZone} and
 * {@code setZone} to get and set the element.</li>
 * </ul>
 *
 * {@code EPPRegistryUpdateResp} is the concrete {@code EPPReponse} associated
 * with {@code EPPRegistryUpdateCmd}.
 */
public class EPPRegistryUpdateCmd extends EPPUpdateCmd {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryUpdateCmd.class);

	/**
	 * XML local name for {@code EPPRegistryUpdateCmd}.
	 */
	public static final String ELM_LOCALNAME = "update";

	/**
	 * XML root tag for {@code EPPRegistryUpdateCmd}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Instance of {@link EPPRegistryZone} to update
	 */
	private EPPRegistryZone zone = null;

	/**
	 * Creates an empty EPPRegistryUpdateCmd object. {@code zone} is set to
	 * {@code null}. Use {@code setZone} to set {@code zone} attribute before
	 * sending the update command.
	 */
	public EPPRegistryUpdateCmd() {
	}

	/**
	 * Creates a new EPPRegistryUpdateCmd object that will update a registry
	 * object based on the info in {@code zone}.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aZone
	 *           Instance of {@link EPPRegistryZone} to update
	 */
	public EPPRegistryUpdateCmd(String aTransId, EPPRegistryZone aZone) {
		super(aTransId);
		this.zone = aZone;
	}

	/**
	 * Get the EPP command Namespace associated with EPPRegistryUpdateCmd.
	 *
	 * @return {@code EPPRegistryMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Validate the state of the {@code EPPRegistryUpdateCmd} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the {@code EPPCodecException} will contain a
	 * description of the error. throws EPPCodecException State error. This will
	 * contain the name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	void validateState() throws EPPCodecException {
		if (this.zone == null) {
			throw new EPPCodecException("zone required attribute is not set");
		}
	}

	/**
	 * Compare an instance of {@code EPPRegistryUpdateCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryUpdateCmd)) {
			cat.error(
			      "EPPRegistryUpdateCmd.equals(): " + aObject.getClass().getName() + " not EPPRegistryUpdateCmd instance");

			return false;
		}
		if (!super.equals(aObject)) {
			cat.error("EPPRegistryUpdateCmd.equals(): super class not equal");

			return false;
		}

		EPPRegistryUpdateCmd theComp = (EPPRegistryUpdateCmd) aObject;

		// Zone
		if (!EqualityUtil.equals(this.zone, theComp.zone)) {
			cat.error("EPPRegistryUpdateCmd.equals(): zone not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryUpdateCmd}.
	 *
	 * @return clone of {@code EPPRegistryUpdateCmd}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryUpdateCmd clone = (EPPRegistryUpdateCmd) super.clone();

		if (this.zone != null) {
			clone.zone = (EPPRegistryZone) this.zone.clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryUpdateCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPRegistryUpdateCmd}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryUpdateCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		try {
			// Validate States
			validateState();
		}
		catch (EPPCodecException e) {
			cat.error("EPPRegistryUpdateCmd.doEncode(): Invalid state on encode: " + e);
			throw new EPPEncodeException("EPPRegistryUpdateCmd invalid state: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Zone
		EPPUtil.encodeComp(aDocument, root, this.zone);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryUpdateCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryUpdateCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {

		// Zone
		this.zone = (EPPRegistryZone) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryZone.ELM_NAME, EPPRegistryZone.class);
	}

	/**
	 * Gets the zone to update.
	 *
	 * @return Instance of {@link EPPRegistryZone} to update
	 */
	public EPPRegistryZone getZone() {
		return this.zone;
	}

	/**
	 * Sets the zone to update.
	 *
	 * @param aZone
	 *           Instance of {@link EPPRegistryZone} to update
	 */
	public void setZone(EPPRegistryZone aZone) {
		this.zone = aZone;
	}
}