/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryMaxSig} defines the maximum signature lifetime policies.
 */
public class EPPRegistryMaxSig implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryMaxSig.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryMaxSig}.
	 */
	public static final String ELM_LOCALNAME = "maxSigLife";

	/**
	 * XML root tag for {@code EPPRegistryMaxSig}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code clientDefined} attribute.
	 */
	public static final String ELM_CLIENT_DEFINED = "clientDefined";

	/**
	 * XML Element Name of {@code defaultLife} attribute.
	 */
	public static final String ELM_DEFAULT = "default";

	/**
	 * XML Element Name of {@code min} attribute.
	 */
	public static final String ELM_MIN = "min";

	/**
	 * XML Element Name of {@code max} attribute.
	 */
	public static final String ELM_MAX = "max";

	/**
	 * An OPTIONAL boolean flag indicating whether the client can set the maximum
	 * signature lifetime with a default value of "false".
	 */
	private Boolean clientDefined = Boolean.FALSE;

	/**
	 * The OPTIONAL default maximum signature lifetime set by the server.
	 */
	private Integer defaultLife;

	/**
	 * An OPTIONAL minimum signature lifetime supported. The &lt;registry:min&gt;
	 * element MUST NOT be defined if the &lt;registry:clientDefined&gt; element
	 * value is "false".
	 */
	private Integer min = null;

	/**
	 * An OPTIONAL maximum signature lifetime supported. The &lt;registry:max&gt;
	 * element MUST NOT be defined if the &lt;registry:clientDefined&gt; element
	 * value is "false". The &lt;registry:max&gt; element MUST NOT be less than
	 * the &lt;registry:min&gt; element.
	 */
	private Integer max = null;

	/**
	 * Default constructor.
	 */
	public EPPRegistryMaxSig() {
	}

	/**
	 * Constructor that takes all attributes as parameters.
	 * 
	 * @param aClientDefined
	 *           Boolean flag indicating whether the client can set the maximum
	 *           signature lifetime. If set to {@code null},
	 *           {@link Boolean#FALSE} will be used.
	 * @param aDefaultLife
	 *           Default maximum signature lifetime set by the server. Set to
	 *           {@code null} if undefined.
	 * @param aMin
	 *           Minimum signature lifetime supported. Set to {@code null} if
	 *           undefined.
	 * @param aMax
	 *           Maximum signature lifetime supported. Set to {@code null} if
	 *           undefined.
	 */
	public EPPRegistryMaxSig(Boolean aClientDefined, Integer aDefaultLife, Integer aMin, Integer aMax) {
		this.setClientDefined(aClientDefined);
		this.defaultLife = aDefaultLife;
		this.min = aMin;
		this.max = aMax;
	}

	/**
	 * Constructor that takes all attributes as parameters.
	 * 
	 * @param aClientDefined
	 *           Boolean flag indicating whether the client can set the maximum
	 *           signature lifetime. Must be set to {@link Boolean#FALSE} for
	 *           this constructor, since the minimum and maximum signature
	 *           lifetime attributes will be defined.
	 * @param aDefaultLife
	 *           Default maximum signature lifetime set by the server.
	 * @param aMin
	 *           Minimum signature lifetime supported.
	 * @param aMax
	 *           Maximum signature lifetime supported.
	 */
	public EPPRegistryMaxSig(boolean clientDefined, int defaultLife, int min, int max) {
		this(Boolean.valueOf(clientDefined), Integer.valueOf(defaultLife), Integer.valueOf(min), Integer.valueOf(max));
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryMaxSig} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryMaxSig} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryMaxSig} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryMaxSig.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Client Defined
		if (this.clientDefined == null) {
			this.clientDefined = Boolean.FALSE;
		}
		EPPUtil.encodeString(aDocument, root, this.clientDefined.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_CLIENT_DEFINED);

		// Default Life
		if (this.defaultLife != null) {
			EPPUtil.encodeString(aDocument, root, this.defaultLife.toString(), EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_DEFAULT);
		}

		// Min
		if (this.min != null) {
			EPPUtil.encodeString(aDocument, root, this.min.toString(), EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MIN);
		}

		// Max
		if (this.max != null) {
			EPPUtil.encodeString(aDocument, root, this.max.toString(), EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MAX);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryMaxSig} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryMaxSig} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Client Defined
		this.clientDefined = EPPUtil.decodeBoolean(aElement, EPPRegistryMapFactory.NS, ELM_CLIENT_DEFINED);

		// Default Life
		this.defaultLife = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_DEFAULT);

		// Min
		this.min = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MIN);

		// Max
		this.max = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MAX);
	}

	/**
	 * Validate the state of the {@code EPPRegistryMaxSig} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	void validateState() throws EPPCodecException {
		// Client Defined
		if (this.clientDefined == null || this.clientDefined == Boolean.FALSE) {
			if (this.min != null || this.max != null) {
				throw new EPPCodecException("None of min or max should be set when clientDefined is set to false");
			}
		}

		// Min
		if (this.min != null && this.min.intValue() < 1) {
			throw new EPPCodecException("min, if specified, should be no less than 1");
		}

		// Max
		if (this.max != null && this.max.intValue() < 1) {
			throw new EPPCodecException("max, if specified, should be no less than 1");
		}

		// Default Life
		if (this.defaultLife != null && this.defaultLife.intValue() < 1) {
			throw new EPPCodecException("defaultLife, if specified, should be no less than 1");
		}

		// Max not less than Min
		if (this.min != null && this.max != null) {
			if (this.max.intValue() < this.min.intValue()) {
				throw new EPPCodecException("max should be no less than min");
			}
		}
	}

	/**
	 * Clone {@code EPPRegistryMaxSig}.
	 *
	 * @return clone of {@code EPPRegistryMaxSig}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryMaxSig} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryMaxSig} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryMaxSig)) {
			return false;
		}

		EPPRegistryMaxSig theComp = (EPPRegistryMaxSig) aObject;

		// Client Defined
		if (!EqualityUtil.equals(this.clientDefined, theComp.clientDefined)) {
			cat.error("EPPRegistryMaxSig.equals(): clientDefined not equal");
			return false;
		}

		// Default Life
		if (!EqualityUtil.equals(this.defaultLife, theComp.defaultLife)) {
			cat.error("EPPRegistryMaxSig.equals(): defaultLife not equal");
			return false;
		}

		// Min
		if (!EqualityUtil.equals(this.min, theComp.min)) {
			cat.error("EPPRegistryMaxSig.equals(): min not equal");
			return false;
		}

		// Max
		if (!EqualityUtil.equals(this.max, theComp.max)) {
			cat.error("EPPRegistryMaxSig.equals(): max not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the OPTIONAL boolean flag indicating whether the client can set the
	 * maximum signature lifetime with a default value of "false".
	 * 
	 * @return Boolean flag indicating whether the client can set the maximum
	 *         signature lifetime with a default value {@link Boolean#FALSE}.
	 */
	public Boolean getClientDefined() {
		return this.clientDefined;
	}

	/**
	 * Sets the OPTIONAL boolean flag indicating whether the client can set the
	 * maximum signature lifetime with a default value of "false".
	 * 
	 * @param aClientDefined
	 *           Boolean flag indicating whether the client can set the maximum
	 *           signature lifetime. If set to {@code null} the default value of
	 *           {@link Boolean#FALSE} will be used.
	 */
	public void setClientDefined(Boolean aClientDefined) {
		if (aClientDefined == null) {
			this.clientDefined = Boolean.FALSE;
		}
		else {
			this.clientDefined = aClientDefined;
		}
	}

	/**
	 * Is the default lifetime defined?
	 *
	 * @return {@code true} if the default lifetime is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasDefaultLife() {
		return (this.defaultLife != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL default maximum signature lifetime set by the server.
	 * 
	 * @return Default maximum signature lifetime set by the server. if defined;
	 *         {@code null} otherwise.
	 */
	public Integer getDefaultLife() {
		return this.defaultLife;
	}

	/**
	 * Sets the OPTIONAL default maximum signature lifetime set by the server.
	 * 
	 * @param aDefaultLife
	 *           Default maximum signature lifetime set by the server. Set to
	 *           {@code null} to clear the attribute.
	 */
	public void setDefaultLife(Integer aDefaultLife) {
		this.defaultLife = aDefaultLife;
	}

	/**
	 * Is the minimum signature lifetime defined?
	 *
	 * @return {@code true} if the minimum signature lifetime is defined;
	 *         {@code false} otherwise.
	 */
	public boolean hasMin() {
		return (this.min != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL minimum signature lifetime supported.
	 * 
	 * @return Minimum signature lifetime supported if defined; {@code null}
	 *         otherwise.
	 */
	public Integer getMin() {
		return this.min;
	}

	/**
	 * Sets the OPTIONAL minimum signature lifetime supported.
	 * 
	 * @param aMin
	 *           Minimum signature lifetime supported. Set to {@code null} to
	 *           clear the attribute.
	 */
	public void setMin(Integer aMin) {
		this.min = aMin;
	}

	/**
	 * Is the maximum signature lifetime defined?
	 *
	 * @return {@code true} if the maximum signature lifetime is defined;
	 *         {@code false} otherwise.
	 */
	public boolean hasMax() {
		return (this.max != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL maximum signature lifetime supported.
	 * 
	 * @return Maximum signature lifetime supported if defined; {@code null}
	 *         otherwise.
	 */
	public Integer getMax() {
		return this.max;
	}

	/**
	 * Sets the OPTIONAL maximum signature lifetime supported.
	 * 
	 * @param aMax
	 *           Maximum signature lifetime supported. Set to {@code null} to
	 *           clear the attribute.
	 */
	public void setMax(Integer aMax) {
		this.max = aMax;
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
