/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPInfoCmd;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents an EPP Registry &lt;info&gt; command that is used to retrieve
 * information associated with a registry. The &lt;registry:info&gt; element
 * MUST contain one of the following child elements:<br>
 * <br>
 *
 * <ul>
 * <li>A &lt;registry:all&gt; empty element that specifies whether or not to
 * query a list of all supported zone objects by the server. Use {@code isAll}
 * and {@code setAll} to get and set the element.</li>
 * <li>A &lt;registry:name&gt; element that contains the fully qualified zone
 * object name for which information is requested. Use {@code getName} and
 * {@code setName} to get and set the element.</li>
 * <li>A &lt;registry:system&gt; Element that is empty and that indicates that
 * the registry system attributes, like maximum connections and timeouts, are
 * queried. Use {@code isSystem} and {@code setSystem} to get and set the
 * element.</li>
 * </ul>
 *
 * A valid {@code EPPRegistryInfoCmd} must contains one and only one of the
 * above elements. <br>
 *
 * <br>
 * {@code EPPRegistryInfoResp} is the concrete {@code EPPReponse} associated
 * with {@code EPPRegistryInfoResp}. <br>
 * <br>
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryInfoResp
 */
public class EPPRegistryInfoCmd extends EPPInfoCmd {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryInfoCmd.class);

	/**
	 * XML local name for {@code EPPRegistryInfoCmd}.
	 */
	public static final String ELM_LOCALNAME = "info";

	/**
	 * XML root tag for {@code EPPRegistryInfoCmd}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name for the {@code all} attribute.
	 */
	private final static String ELM_REGISTRY_ALL = "all";

	/**
	 * XML Element Name for the {@code system} attribute.
	 */
	private final static String ELM_REGISTRY_SYSTEM = "system";

	/**
	 * Setting this attribute to {@code true} to query a list of all of the
	 * supported zone objects from the server. This CANNOT be used together with
	 * {@code name}. If {@code name} is not empty or {@code system} is
	 * {@code true}, then this attribute must be set to {@code false}..
	 */
	private boolean all = false;

	/**
	 * Fully qualified name of zone object to get information on. Cannot be used
	 * together with {@code all}. If {@code system} is {@code true} or
	 * {@code all} is {@code true}, then this attribute must be set to
	 * {@code false}..
	 */
	private EPPRegistryZoneName name;

	/**
	 * Setting this attribute to {@code true} to query registry system attributes
	 * from the server. This CANNOT be used together with {@code name} or
	 * {@code all}. If {@code name} is not empty or {@code all} is {@code true},
	 * then this attribute must be set to {@code false}.
	 */
	private boolean system = false;

	/**
	 * {@code EPPRegistryInfoCmd} default constructor. The {@code all} is set to
	 * {@code false} and the {@code name} is set to {@code null}. Either
	 * {@code all} or {@code name} must be set before invoking {@code encode}.
	 */
	public EPPRegistryInfoCmd() {
	}

	/**
	 * {@code EPPRegistryInfoCmd} constructor that takes the qualified aLabel
	 * zone object name as an argument. Attribute {@code all} and attribute
	 * {@code system} are set to {@code false} .
	 *
	 * @param aTransId
	 *           transaction Id associated with command
	 * @param aName
	 *           fully qualified aLabel zone object name to get information on
	 */
	public EPPRegistryInfoCmd(String aTransId, String aName) {
		super(aTransId);

		this.setName(aName);
		this.all = false;
		this.system = false;
	}

	/**
	 * {@code EPPRegistryInfoCmd} constructor that takes the qualified 
	 * zone object name as an argument. Attribute {@code all} and attribute
	 * {@code system} are set to {@code false} .
	 *
	 * @param aTransId
	 *           transaction Id associated with command
	 * @param aName
	 *           fully qualified zone object name to get information on
	 */
	public EPPRegistryInfoCmd(String aTransId, EPPRegistryZoneName aName) {
		super(aTransId);

		this.setName(aName);
		this.all = false;
		this.system = false;
	}
	
	
	/**
	 * {@code EPPRegistryInfoCmd} constructor that tries to query a list of all
	 * supported zone objects from the server. Attribute {@code name} is set to
	 * {@code null} and {@code system} is set to {@code false}.
	 *
	 * @param aTransId
	 *           transaction Id associated with command
	 * @param aAll
	 *           Boolean value for the {@code all} attribute.
	 */
	public EPPRegistryInfoCmd(String aTransId, boolean aAll) {
		super(aTransId);

		this.name = null;
		this.all = aAll;
		this.system = false;
	}

	/**
	 * {@code EPPRegistryInfoCmd} constructor that tries to query a list of all
	 * supported zone objects from the server if {@code aAll} is {@code true} or
	 * query for the registry system attributes if {@code aSystem} is
	 * {@code true}.
	 *
	 * @param aTransId
	 *           transaction Id associated with command
	 * @param aAll
	 *           Boolean value for the {@code all} attribute.
	 * @param aSystem
	 *           Boolean value for the {@code system} attribute.
	 */
	public EPPRegistryInfoCmd(String aTransId, boolean aAll, boolean aSystem) {
		super(aTransId);

		this.name = null;
		this.all = aAll;
		this.system = aSystem;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryInfoCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPRegistryInfoCmd}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryInfoCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		// Validate state
		if (this.name == null && !this.all && !this.system) {
			throw new EPPEncodeException("At least one needs to be set (name, all, or system)");
		}
		if (this.name != null && (this.all || this.system)) {
			throw new EPPEncodeException("Name cannot be set with either all or system set");
		}
		if (this.all && this.system) {
			throw new EPPEncodeException("Both all and system cannot be set");
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Name?
		if (this.hasName()) {
			EPPUtil.encodeComp(aDocument, root, this.name);
		} // All?
		else if (this.all) {
			root.appendChild(aDocument.createElementNS(EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_REGISTRY_ALL));
		} // System
		else {
			root.appendChild(aDocument.createElementNS(EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_REGISTRY_SYSTEM));
		}

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryInfoCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryInfoCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = (EPPRegistryZoneName) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryZoneName.ELM_ZONE_NAME, EPPRegistryZoneName.class);

		// All
		this.all = EPPUtil.getElementByTagNameNS(aElement, EPPRegistryMapFactory.NS, ELM_REGISTRY_ALL) != null;

		// System
		this.system = EPPUtil.getElementByTagNameNS(aElement, EPPRegistryMapFactory.NS, ELM_REGISTRY_SYSTEM) != null;

		return;
	}

	/**
	 * Gets the EPP command namespace associated with {@code EPPRegistryInfoCmd}
	 * .
	 *
	 * @return {@code EPPRegistryMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Compare an instance of {@code EPPRegistryInfoCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryInfoCmd)) {
			cat.error("EPPRegistryInfoCmd.equals(): " + aObject.getClass().getName() + " not EPPRegistryInfoCmd instance");

			return false;
		}

		if (!super.equals(aObject)) {
			cat.error("EPPRegistryInfoCmd.equals(): super class not equal");

			return false;
		}

		EPPRegistryInfoCmd theComp = (EPPRegistryInfoCmd) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryInfoCmd.equals(): name not equal");
			return false;
		}

		// All
		if (this.all != theComp.all) {
			cat.error("EPPRegistryInfoCmd.equals(): all not equal");
			return false;
		}

		// System
		if (this.system != theComp.system) {
			cat.error("EPPRegistryInfoCmd.equals(): system not equal");
			return false;
		}

		return true;
	}
	
	/**
	 * Clone {@code EPPRegistryInfoCmd}.
	 * 
	 * @return clone of {@code EPPRegistryInfoCmd}
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryInfoCmd clone = (EPPRegistryInfoCmd) super.clone();

		// Name
		if (this.name != null) {
			clone.name = (EPPRegistryZoneName) this.name.clone();
		}

		return clone;
	}
	

	/**
	 * Is the name defined?
	 *
	 * @return {@code true} if the name is defined; {@code false} otherwise.
	 */
	public boolean hasName() {
		return (this.name != null ? true : false);
	}

	/**
	 * Gets the zone name.
	 *
	 * @return Zone name if set; {@code null} otherwise.
	 */
	public EPPRegistryZoneName getName() {
		return this.name;
	}

	/**
	 * Sets the zone name
	 *
	 * @param aName
	 *           Zone name
	 */
	public void setName(EPPRegistryZoneName aName) {
		this.name = aName;
	}

	/**
	 * Sets the aLabel zone name
	 *
	 * @param aName
	 *           aLabel zone name
	 */
	public void setName(String aName) {
		this.name = new EPPRegistryZoneName(aName, EPPRegistryZoneName.Form.aLabel);
	}

	/**
	 * Get the flag to query all supported zone objects.
	 *
	 * @return {@code true} if client intends to query a list of all supported
	 *         zone object; {@code false} otherwise.
	 */
	public boolean isAll() {
		return this.all;
	}

	/**
	 * Set the flag to query all supported zone objects.
	 *
	 * @param aAll
	 *           {@code true} if client intends to query a list of all supported
	 *           zone objects; {@code false} otherwise.
	 */
	public void setAll(boolean aAll) {
		this.all = aAll;
	}

	/**
	 * Get the flag to query the registry system attributes.
	 *
	 * @return {@code true} if client intends to query the registry system
	 *         attributes; {@code false} otherwise.
	 */
	public boolean isSystem() {
		return this.system;
	}

	/**
	 * Set the flag to query the registry system attributes.
	 *
	 * @param aSystem
	 *           {@code true} if client intends to query a list of all supported
	 *           zone objects; {@code false} otherwise.
	 */
	public void setSystem(boolean aSystem) {
		this.system = aSystem;
	}

}
