/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * The minimum and maximum length and the minimum and maximum number of the
 * &lt;contact:street&gt; elements defined in [RFC5733].
 */
public class EPPRegistryContactStreet extends EPPRegistryAbstractMinMax {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryContactStreet.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryContactStreet}.
	 */
	public static final String ELM_LOCALNAME = "street";

	/**
	 * XML root tag for {@code EPPRegistryContactStreet}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code minEntry} attribute.
	 */
	public static final String ELM_MIN_ENTRY = "minEntry";

	/**
	 * XML Element Name of {@code maxEntry} attribute.
	 */
	public static final String ELM_MAX_ENTRY = "maxEntry";

	/**
	 * The minimum number of &lt;contact:street&gt; elements.
	 */
	private Integer minEntry = null;

	/**
	 * The maximum number of &lt;contact:street&gt; elements.
	 */
	private Integer maxEntry = null;

	/**
	 * Creates a new {@code EPPRegistryContactStreet} object.
	 */
	public EPPRegistryContactStreet() {
	}

	/**
	 * Creates a new {@code EPPRegistryContactStreet} object that takes both the
	 * minimum and maximum values as {@code Integer}'s.
	 *
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 * @param aMinEntry
	 *           Minimum number of &lt;contact:street&gt; elements.
	 * @param aMaxEntry
	 *           Maximum number of &lt;contact:street&gt; elements.
	 */
	public EPPRegistryContactStreet(Integer aMin, Integer aMax, Integer aMinEntry, Integer aMaxEntry) {
		super(aMin, aMax);
		this.minEntry = aMinEntry;
		this.maxEntry = aMaxEntry;
	}

	/**
	 * Creates a new {@code EPPRegistryContactStreet} object that takes both the
	 * minimum and maximum values as {@code int}'s.
	 *
	 * @param aMin
	 *           Minimum value
	 * @param aMax
	 *           Maximum value
	 * @param aMinEntry
	 *           Minimum number of &lt;contact:street&gt; elements.
	 * @param aMaxEntry
	 *           Maximum number of &lt;contact:street&gt; elements.
	 */
	public EPPRegistryContactStreet(int aMin, int aMax, int aMinEntry, int aMaxEntry) {
		super(aMin, aMax);
		this.minEntry = Integer.valueOf(aMinEntry);
		this.maxEntry = Integer.valueOf(aMaxEntry);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryContactStreet} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryContactStreet} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryContactStreet}
	 *               instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		validateState();

		Element root = super.encode(aDocument);

		// Min Entry
		EPPUtil.encodeString(aDocument, root, this.minEntry.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MIN_ENTRY);

		// Max Entry
		EPPUtil.encodeString(aDocument, root, this.maxEntry.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MAX_ENTRY);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryContactStreet} attributes from the aElement
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryContactStreet}
	 *           from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		super.decode(aElement);

		// Min Entry
		this.minEntry = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MIN_ENTRY);

		// Max Entry
		this.maxEntry = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MAX_ENTRY);
	}

	/**
	 * Clone {@code EPPRegistryContactStreet}.
	 *
	 * @return clone of {@code EPPRegistryContactStreet}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryContactStreet} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryContactStreet} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!super.equals(aObject)) {
			return false;
		}

		if (!(aObject instanceof EPPRegistryContactStreet)) {
			return false;
		}

		EPPRegistryContactStreet theComp = (EPPRegistryContactStreet) aObject;

		// MinEntry
		if (!EqualityUtil.equals(this.minEntry, theComp.minEntry)) {
			cat.error("EPPRegistryContactStreet.equals(): minEntry not equal");
			return false;
		}

		// MaxEntry
		if (!EqualityUtil.equals(this.maxEntry, theComp.maxEntry)) {
			cat.error("EPPRegistryContactStreet.equals(): maxEntry not equal");
			return false;
		}

		return true;
	}

	/**
	 * Gets the minimum entry value, which defines the minimum number of
	 * &lt;contact:street&gt; elements.
	 *
	 * @return Minimum entry value if defined; {@code null} otherwise.
	 */
	public Integer getMinEntry() {
		return this.minEntry;
	}

	/**
	 * Sets the minimum entry value, which defines the minimum number of
	 * &lt;contact:street&gt; elements.
	 *
	 * @param aMinEntry
	 *           Minimum number of &lt;contact:street&gt; elements.
	 */
	public void setMinEntry(Integer aMinEntry) {
		this.minEntry = aMinEntry;
	}

	/**
	 * Gets the maximum entry value, which defines the maximum number of
	 * &lt;contact:street&gt; elements.
	 *
	 * @return Maximum entry value if defined; {@code null} otherwise.
	 */
	public Integer getMaxEntry() {
		return this.maxEntry;
	}

	/**
	 * Sets the maximum entry value, which defines the maximum number of
	 * &lt;contact:street&gt; elements.
	 *
	 * @param aMaxEntry
	 *           Maximum number of &lt;contact:street&gt; elements.
	 */
	public void setMaxEntry(Integer aMaxEntry) {
		this.maxEntry = aMaxEntry;
	}

	/**
	 * Validate the state of the {@code EPPRegistryContactStreet} instance. A
	 * valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the {@code EPPCodecException} will contain a
	 * description of the error.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	@Override
	protected void validateState() throws EPPEncodeException {

		super.validateState();
		if ((this.minEntry == null) || (this.minEntry.intValue() < 0)) {
			throw new EPPEncodeException(
			      "Invalid state on EPPRegistryContactStreet.encode: minEntry is required and should be greater than or equal to 0");
		}

		if ((this.maxEntry == null) || (this.maxEntry.intValue() < this.minEntry.intValue())
		      || this.maxEntry.intValue() > 3) {
			throw new EPPEncodeException(
			      "Invalid state on EPPRegistryContactStreet.encode: maxEntry is required and should be greater than or equal to min, and less than or equal to 3");
		}

	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the root element local name.
	 * 
	 * @return Root element local name.
	 */
	@Override
	protected String getRootName() {
		return ELM_LOCALNAME;
	}

	/**
	 * Gets the minimum element local name.
	 * 
	 * @return Minimum element local name.
	 */
	@Override
	protected String getElmMin() {
		return ELM_MIN_LENGTH;
	}

	/**
	 * Gets the maximum element local name.
	 * 
	 * @return Maximum element local name.
	 */
	@Override
	protected String getElmMax() {
		return ELM_MAX_LENGTH;
	}

	/**
	 * Gets the {@link Logger} to use.
	 * 
	 * @return {@link Logger} instance to use for logging.
	 */
	@Override
	protected Logger getLogger() {
		return cat;
	}

}
