/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryCheckResult} provides the result of checking the
 * availability of an individual zone.
 */
public class EPPRegistryCheckResult implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryCheckResult.class);

	/**
	 * XML local name for {@code EPPRegistryInfoResp}.
	 */
	public static final String ELM_LOCALNAME = "cd";

	/**
	 * XML root tag for {@code EPPRegistryCheckResult}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name for the {@code reason} attribute.
	 */
	private final static String ELM_REGISTRY_REASON = "reason";

	/**
	 * XML attribute name for the {@code known} attribute.
	 */
	private final static String ATTR_AVAIL = "avail";

	/**
	 * XML attribute name for the {@code registry:reason} attribute.
	 */
	private final static String ATTR_LANG = "lang";

	/**
	 * Default XML attribute value for registry reason language.
	 */
	private final static String VALUE_LANG = "en";

	/**
	 * XML attribute value for the {@code lang} attribute.
	 */
	private String language = VALUE_LANG;

	/**
	 * Zone Name associated with result.
	 */
	private EPPRegistryZoneName name;

	/**
	 * Is the zone Name (name) available?
	 */
	private Boolean available;

	/**
	 * Zone unavailable reason.
	 */
	private String reason;

	/**
	 * Creates a new {@code EPPRegistryCheckResult} object with an empty zone
	 * name and availability set to {@code true}. The Zone name must be set via
	 * {@link #setName(String)} prior to calling {@link #encode(Document)}.
	 */
	public EPPRegistryCheckResult() {
		this.name = null;
		this.available = Boolean.TRUE;
	}

	/**
	 * Creates a new {@code EPPRegistryCheckResult} object with a defined aLabel
	 * zone name and available value.
	 *
	 * @param aName
	 *           aLabel zone name
	 * @param aAvailable
	 *           {@code true} if available; {@code false} otherwise.
	 */
	public EPPRegistryCheckResult(String aName, Boolean aAvailable) {
		this.setName(aName);
		this.available = aAvailable;
	}

	/**
	 * Creates a new {@code EPPRegistryCheckResult} object with a defined zone
	 * name and available value.
	 *
	 * @param aName
	 *           Zone name
	 * @param aAvailable
	 *           {@code true} if available; {@code false} otherwise.
	 */
	public EPPRegistryCheckResult(EPPRegistryZoneName aName, Boolean aAvailable) {
		this.setName(aName);
		this.available = aAvailable;
	}

	/**
	 * Creates a new {@code EPPRegistryCheckResult} object with a defined aLabel
	 * zone name, available value, and reason.
	 *
	 * @param aName
	 *           aLabel zone name
	 * @param aAvailable
	 *           Should be set to {@code false} for this constructor.
	 * @param aReason
	 *           Reason Unavailable reason using the default language of
	 *           {@link #VALUE_LANG}.
	 */
	public EPPRegistryCheckResult(String aName, Boolean aAvailable, String aReason) {
		this.setName(aName);
		this.available = aAvailable;
		this.reason = aReason;
	}

	/**
	 * Creates a new {@code EPPRegistryCheckResult} object with a defined zone
	 * name, available value, and reason.
	 *
	 * @param aName
	 *           Zone name
	 * @param aAvailable
	 *           Should be set to {@code false} for this constructor.
	 * @param aReason
	 *           Reason Unavailable reason using the default language of
	 *           {@link #VALUE_LANG}.
	 */
	public EPPRegistryCheckResult(EPPRegistryZoneName aName, Boolean aAvailable, String aReason) {
		this.setName(aName);
		this.available = aAvailable;
		this.reason = aReason;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryCheckResult} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryCheckResult} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryCheckResult} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Validate state
		if (this.name == null) {
			throw new EPPEncodeException("Zone name required element is not set");
		}
		if (this.available == null) {
			throw new EPPEncodeException("Zone available required attribute is not set");
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Zone Name
		Element nameElm = this.name.encode(aDocument);
		root.appendChild(nameElm);

		// Available
		EPPUtil.encodeBooleanAttr(nameElm, ATTR_AVAIL, this.available.booleanValue());

		// Unavailable Reason
		if (this.reason != null) {
			Element reasonElm = aDocument.createElementNS(EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_REGISTRY_REASON);
			root.appendChild(reasonElm);

			// Language
			if (!this.language.equals(VALUE_LANG)) {
				reasonElm.setAttribute(ATTR_LANG, this.language);
			}

			// Unavailable Reason
			Text aReason = aDocument.createTextNode(this.reason);
			reasonElm.appendChild(aReason);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryCheckResult} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryCheckResult} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Zone Name and Available
		Element theNameElm = EPPUtil.getElementByTagNameNS(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryZoneName.ELM_ZONE_NAME);
		this.name = new EPPRegistryZoneName();
		this.name.decode(theNameElm);
		this.available = Boolean.valueOf(EPPUtil.decodeBooleanAttr(theNameElm, ATTR_AVAIL));

		// Unavailable Reason
		Element theReasonElm = EPPUtil.getElementByTagNameNS(aElement, EPPRegistryMapFactory.NS, ELM_REGISTRY_REASON);

		if (theReasonElm != null) {
			this.reason = theReasonElm.getFirstChild().getNodeValue();

			String theLang = theReasonElm.getAttribute(ATTR_LANG);

			if (theLang.length() > 0) {
				if (!theLang.equals(VALUE_LANG)) {
					setLanguage(theLang);
				}
			}
		}
	}

	/**
	 * Clone {@code EPPRegistryCheckResult}.
	 *
	 * @return clone of {@code EPPRegistryCheckResult}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryCheckResult clone = null;

		clone = (EPPRegistryCheckResult) super.clone();

		// Zone Name
		if (this.name != null) {
			clone.name = (EPPRegistryZoneName) this.name.clone();
		}

		return clone;
	}

	/**
	 * Compare an instance of {@code EPPRegistryCheckResult} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryCheckResult)) {
			return false;
		}

		EPPRegistryCheckResult theComp = (EPPRegistryCheckResult) aObject;

		// Zone Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryCheckResult.equals(): name not equal");
			return false;
		}

		// Available
		if (!EqualityUtil.equals(this.available, theComp.available)) {
			cat.error("EPPRegistryCheckResult.equals(): available not equal");
			return false;
		}

		// Language
		if (!EqualityUtil.equals(this.language, theComp.language)) {
			cat.error("EPPRegistryCheckResult.equals(): language not equal");
			return false;
		}

		// Unavailable Reason
		if (!EqualityUtil.equals(this.reason, theComp.reason)) {
			cat.error("EPPRegistryCheckResult.equals(): reason not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the unavailable reason language with the default value of
	 * {@link #VALUE_LANG}.
	 *
	 * @return Unavailable reason langauge with the default value of
	 *         {@link #VALUE_LANG}.
	 */
	public String getLanguage() {
		return this.language;
	}

	/**
	 * Sets the unavailable reason language with the default value of
	 * {@link #VALUE_LANG}.
	 *
	 * @param aLanguage
	 *           Language of the unavailable reason set with
	 *           {@link #setReason(String)}.
	 */
	public void setLanguage(String aLanguage) {
		this.language = aLanguage;
	}

	/**
	 * Gets the zone name.
	 *
	 * @return Zone name if set; {@code null} otherwise.
	 */
	public EPPRegistryZoneName getName() {
		return this.name;
	}

	/**
	 * Sets the zone name
	 *
	 * @param aName
	 *           Zone name
	 */
	public void setName(EPPRegistryZoneName aName) {
		this.name = aName;
	}

	/**
	 * Sets the aLabel zone name
	 *
	 * @param aName
	 *           aLabel zone name
	 */
	public void setName(String aName) {
		this.name = new EPPRegistryZoneName(aName, EPPRegistryZoneName.Form.aLabel);
	}

	/**
	 * Gets the zone available.
	 *
	 * @return {@code true} if available; {@code false} otherwise.
	 */
	public Boolean isAvailable() {
		return this.available;
	}

	/**
	 * Sets the zone available.
	 *
	 * @param aAvailable
	 *           {@code true} if available; {@code false} otherwise.
	 */
	public void setAvailable(Boolean aAvailable) {
		this.available = aAvailable;
	}

	/**
	 * Gets the unavailable reason. This value should be {@code null} if
	 * {@link #isAvailable()} is {@code true}.
	 *
	 * @return The unavailable reason if set; {@code null} otherwise.
	 */
	public String getReason() {
		return this.reason;
	}

	/**
	 * Sets the unavailable reason. This value should be {@code null} if
	 * {@link #isAvailable()} is {@code true}.
	 *
	 * @param aReason
	 *           Unavailable reason. Set to {@code null} if zone is available.
	 */
	public void setReason(String aReason) {
		this.reason = aReason;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
