/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryBatchJob} contains the information for an individual batch
 * job.
 */
public class EPPRegistryBatchJob implements EPPCodecComponent {
	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryBatchJob.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * The default batch job schedule time zone.
	 */
	public static final String DEFAULT_SCHEDULE_TZ = "UTC";

	/**
	 * XML local name for {@code EPPRegistryBatchJob}.
	 */
	public static final String ELM_LOCALNAME = "batchJob";

	/**
	 * XML root tag for {@code EPPRegistryBatchJob}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the {@code name} attribute.
	 */
	final static String ELM_JOB_NAME = "name";

	/**
	 * XML local name for the {@code description} attribute.
	 */
	final static String ELM_DESCRIPTION = "description";

	/**
	 * XML local name for the {@code schedule} attribute.
	 */
	final static String ELM_SCHEDULE = "schedule";

	/**
	 * XML attribute name for the {@code scheduleTimeZone} attribute.
	 */
	final static String ATTR_TZ = "tz";

	/**
	 * Name of the batch job, like "autoRenew" or "pendingDelete".
	 */
	private String name = null;

	/**
	 * OPTIONAL free-form description of batch job, like "Auto Renew Batch" or
	 * "Pending Delete Batch".
	 */
	private String description = null;

	/**
	 * Execution schedule for the batch job based on the cron schedule format.
	 */
	private String schedule = null;

	/**
	 * Time zone for the schedule with the default of UTC.
	 */
	private String scheduleTimeZone = DEFAULT_SCHEDULE_TZ;

	/**
	 * Default constructor for {@code EPPRegistryBatchJob}.
	 */
	public EPPRegistryBatchJob() {
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that takes all required
	 * attributes.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aSchedule
	 *           Execution schedule for the batch job based on the cron schedule
	 *           format.
	 */
	public EPPRegistryBatchJob(String aName, String aSchedule) {
		this.name = aName;
		this.schedule = aSchedule;
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that takes all attributes.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aSchedule
	 *           Execution schedule for the batch job based on the cron schedule
	 *           format.
	 * @param aDescription
	 *           OPTIONAL free-form description of batch job, like
	 *           "Auto Renew Batch" or "Pending Delete Batch".. Set to
	 *           {@code null} if undefined.
	 */
	public EPPRegistryBatchJob(String aName, String aSchedule, String aDescription) {
		this(aName, aSchedule);
		this.description = aDescription;
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that takes all attributes.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aSchedule
	 *           Execution schedule for the batch job based on the cron schedule
	 *           format.
	 * @param aDescription
	 *           OPTIONAL free-form description of batch job, like
	 *           "Auto Renew Batch" or "Pending Delete Batch".. Set to
	 *           {@code null} if undefined.
	 * @param aScheduleTimeZone the schedule time zone to set. If set to {@code null} the
	 *           default of {@link #DEFAULT_SCHEDULE_TZ} will be used.
	 */
	public EPPRegistryBatchJob(String aName, String aSchedule, String aDescription, String aScheduleTimeZone) {
		this(aName, aSchedule, aDescription);
		this.setScheduleTimeZone(aScheduleTimeZone);
	}
	
	
	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryBatchJob} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryBatchJob} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryBatchJob} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Validate state
		if (this.name == null) {
			throw new EPPEncodeException("EPPRegistryBatchJob name is null");
		}
		if (this.schedule == null) {
			throw new EPPEncodeException("EPPRegistryBatchJob schedule is null");
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_JOB_NAME);

		// Description
		if (this.description != null) {
			EPPUtil.encodeString(aDocument, root, this.description, EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_DESCRIPTION);
		}

		// Schedule
		Element theScheduleElm = aDocument.createElementNS(EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_SCHEDULE);
		Text theScheduleText = aDocument.createTextNode(String.valueOf(this.schedule));
		theScheduleElm.appendChild(theScheduleText);
		root.appendChild(theScheduleElm);

		// Schedule Time Zone
		theScheduleElm.setAttribute(ATTR_TZ, this.scheduleTimeZone);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryBatchJob} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryBatchJob} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_JOB_NAME);

		// Description
		this.description = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_DESCRIPTION);

		// Schedule
		Element theScheduleElm = EPPUtil.getElementByTagNameNS(aElement, EPPRegistryMapFactory.NS, ELM_SCHEDULE);
		Node theScheduleText = theScheduleElm.getFirstChild();
		if (theScheduleText != null) {
			this.schedule = theScheduleText.getNodeValue();
		}

		// Schedule Time Zone
		this.setScheduleTimeZone(theScheduleElm.getAttribute(ATTR_TZ));
	}

	/**
	 * Clone {@code EPPRegistryBatchJob}.
	 *
	 * @return clone of {@code EPPRegistryBatchJob}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryBatchJob} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryBatchJob} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryBatchJob)) {
			return false;
		}

		EPPRegistryBatchJob theComp = (EPPRegistryBatchJob) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryBatchJob.equals(): name not equal");
			return false;
		}

		// Description
		if (!EqualityUtil.equals(this.description, theComp.description)) {
			cat.error("EPPRegistryBatchJob.equals(): description not equal");
			return false;
		}

		// Schedule
		if (!EqualityUtil.equals(this.schedule, theComp.schedule)) {
			cat.error("EPPRegistryBatchJob.equals(): schedule not equal");
			return false;
		}

		// Schedule Time Zone
		if (!EqualityUtil.equals(this.scheduleTimeZone, theComp.scheduleTimeZone)) {
			cat.error("EPPRegistryBatchJob.equals(): scheduleTimeZone not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Is the name defined?
	 *
	 * @return {@code true} if the name is defined; {@code false} otherwise.
	 */
	public boolean hasName() {
		return (this.name != null ? true : false);
	}

	/**
	 * Gets the name of the batch job, like "autoRenew" or "pendingDelete".
	 *
	 * @return Name of the batch job if defined; {@code null} otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the name of the batch job, like "autoRenew" or "pendingDelete".
	 *
	 * @param aName
	 *           Name of the batch job
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Is the description defined?
	 *
	 * @return {@code true} if the description is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasDescription() {
		return (this.description != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL free-form description of batch job, like
	 * "Auto Renew Batch" or "Pending Delete Batch".
	 *
	 * @return batch job description if defined; {@code null} otherwise.
	 */
	public String getDescription() {
		return this.description;
	}

	/**
	 * Sets the OPTIONAL free-form description of batch job, like
	 * "Auto Renew Batch" or "Pending Delete Batch".
	 *
	 * @param aDescription
	 *           the description to set
	 */
	public void setDescription(String aDescription) {
		this.description = aDescription;
	}

	/**
	 * Gets the execution schedule for the batch job using cron format.
	 *
	 * @return The execution schedule if defined; {@code null} otherwise.
	 */
	public String getSchedule() {
		return this.schedule;
	}

	/**
	 * Sets the execution schedule for the batch job using cron format.
	 *
	 * @param aSchedule
	 *           the schedule to set
	 */
	public void setSchedule(String aSchedule) {
		this.schedule = aSchedule;
	}

	/**
	 * Gets the schedule time zone
	 * 
	 * @return The schedule time zone
	 */
	public String getScheduleTimeZone() {
		return scheduleTimeZone;
	}

	/**
	 * Sets the schedule time zone
	 * 
	 * @param aScheduleTimeZone
	 *           the schedule time zone to set. If set to {@code null} the
	 *           default of {@link #DEFAULT_SCHEDULE_TZ} will be used.
	 */
	public void setScheduleTimeZone(String aScheduleTimeZone) {
		if (aScheduleTimeZone == null) {
			this.scheduleTimeZone = DEFAULT_SCHEDULE_TZ;
		}
		else {
			this.scheduleTimeZone = aScheduleTimeZone;
		}
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
