/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.serverstub;

import org.apache.log4j.Logger;

import com.verisign.epp.codec.gen.EPPLoginCmd;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.serverstub.v01.LoginSecV01GenHandler;
import com.verisign.epp.serverstub.v02.LoginSecV02GenHandler;
import com.verisign.epp.serverstub.v03.LoginSecV03GenHandler;
import com.verisign.epp.serverstub.v04.LoginSecV04GenHandler;
import com.verisign.epp.util.EPPCatFactory;

/**
 * The <code>LoginSecGenHandler</code> class is an extension of the
 * {@link GenHandler} to override the login command based on one of the
 * supported login security extension versions. It handles accepting commands in
 * a server that supports more than one version of the login security extension.
 * The version specific <code>LoginSecSubGenHandler</code> is called based on
 * the login security extension version passed.
 *
 * @see com.verisign.epp.framework.EPPEvent
 * @see com.verisign.epp.framework.EPPEventResponse
 */
public class LoginSecGenHandler extends GenHandler {

	/** Used for logging */
	private static Logger cat = Logger.getLogger(LoginSecGenHandler.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	// Sub-handler for handling version 0.1 extension messages.
	LoginSecV01GenHandler v01SubHandler = new LoginSecV01GenHandler();

	// Sub-handler for handling version 0.2 extension messages.
	LoginSecV02GenHandler v02SubHandler = new LoginSecV02GenHandler();

	// Sub-handler for handling version 0.3 extension messages.
	LoginSecV03GenHandler v03SubHandler = new LoginSecV03GenHandler();

	// Sub-handler for handling version 0.4 extension messages.
	LoginSecV04GenHandler v04SubHandler = new LoginSecV04GenHandler();

	/**
	 * Handle a login command by routing commands with the login security
	 * extension to the appropriate version of handler; otherwise send to the
	 * base handler.
	 *
	 * @param aEvent
	 *           Event must contain an {@link EPPLoginCmd} for handling
	 * @param aData
	 *           Extra server data, which is <code>SessionData</code> for the
	 *           Stub Server.
	 *
	 * @return Response to handling the login command
	 */
	protected EPPEventResponse doLogin(EPPEvent aEvent, Object aData) {
		SessionData theSessionData = (SessionData) aData;
		EPPLoginCmd theCommand = (EPPLoginCmd) aEvent.getMessage();

		// v04 of login security extension?
		if (theCommand.hasExtension(com.verisign.epp.codec.loginsec.v04.EPPLoginSec.class)) {
			return this.v04SubHandler.doLogin(aEvent, aData);
		} // v03 of login security extension?
		else if (theCommand.hasExtension(com.verisign.epp.codec.loginsec.v03.EPPLoginSec.class)) {
			return this.v03SubHandler.doLogin(aEvent, aData);
		} // v02 of login security extension?
		else if (theCommand.hasExtension(com.verisign.epp.codec.loginsec.v02.EPPLoginSec.class)) {
			return this.v02SubHandler.doLogin(aEvent, aData);
		} // v01 of login security extension?
		else if (theCommand.hasExtension(com.verisign.epp.codec.loginsec.v01.EPPLoginSec.class)) {
			return this.v01SubHandler.doLogin(aEvent, aData);
		}
		else {
			// v04 of login security extension supported?
			if (theCommand.hasExtensionService(com.verisign.epp.codec.loginsec.v04.EPPLoginSecExtFactory.NS)) {
				return this.v04SubHandler.doLogin(aEvent, aData);
			} // v03 of login security extension supported?
			if (theCommand.hasExtensionService(com.verisign.epp.codec.loginsec.v03.EPPLoginSecExtFactory.NS)) {
				return this.v03SubHandler.doLogin(aEvent, aData);
			} // v02 of login security extension supported?
			else if (theCommand.hasExtensionService(com.verisign.epp.codec.loginsec.v02.EPPLoginSecExtFactory.NS)) {
				return this.v02SubHandler.doLogin(aEvent, aData);
			} // v01 of login security extension supported?
			else if (theCommand.hasExtensionService(com.verisign.epp.codec.loginsec.v01.EPPLoginSecExtFactory.NS)) {
				return this.v01SubHandler.doLogin(aEvent, aData);
			}
			else {
				cat.info("doLogin: No login security extension, executing default handler");
				return super.doLogin(aEvent, aData);

			}

		}

	}

}
