/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.loginsec.v02;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Login Security Data Extension that extends an
 * {@link com.verisign.epp.codec.gen.EPPResponse} to support passing security
 * event information back to the client.
 */
public class EPPLoginSecData implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPLoginSecData.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for <code>EPPLoginSecData</code>.
	 */
	public static final String ELM_LOCALNAME = "loginSecData";

	/**
	 * XML root tag for <code>EPPLoginSecData</code>.
	 */
	public static final String ELM_NAME = EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Security events to return to the client
	 */
	private List<EPPLoginSecEvent> events = new ArrayList<EPPLoginSecEvent>();

	/**
	 * Default constructor for <code>EPPLoginSecData</code>.
	 */
	public EPPLoginSecData() {
	}

	/**
	 * <code>EPPLoginSecData</code> constructor that takes the list of security
	 * events.
	 * 
	 * @param aEvents
	 *           List of security events.
	 */
	public EPPLoginSecData(List<EPPLoginSecEvent> aEvents) {
		this.events = aEvents;
	}

	/**
	 * Has at least one set security event?
	 * 
	 * @return <code>true</code> if there is at least one security event;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasEvent() {
		if (this.events != null && !this.events.isEmpty()) {
			return true;
		}
		else {
			return false;
		}

	}

	/**
	 * Adds an event to the list of security events.
	 * 
	 * @param aEvent
	 *           Event to add.
	 */
	public void addEvent(EPPLoginSecEvent aEvent) {
		if (aEvent != null) {
			this.events.add(aEvent);
		}
	}

	/**
	 * Gets the list of security events.
	 * 
	 * @return List of {@link EPPLoginSecEvent} instances. The list should never
	 *         be <code>null</code>.
	 */
	public List<EPPLoginSecEvent> getEvents() {
		return this.events;
	}

	/**
	 * Sets the list of security events.
	 * 
	 * @param aEvents
	 *           List of {@link EPPLoginSecEvent} instances.
	 */
	public void setEvents(List<EPPLoginSecEvent> aEvents) {
		if (aEvents == null) {
			this.events = new ArrayList<EPPLoginSecEvent>();
		}
		else {
			this.events = aEvents;
		}
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPLoginSecData</code>
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPLoginSecData.encode(Document)");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPLoginSecExtFactory.NS, ELM_NAME);

		// Events
		EPPUtil.encodeCompList(aDocument, root, this.events);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           <code>Element</code> to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Events
		this.events = EPPUtil.decodeCompList(aElement, EPPLoginSecExtFactory.NS, EPPLoginSecEvent.ELM_NAME,
		      EPPLoginSecEvent.class);
		if (this.events == null) {
			this.events = new ArrayList<EPPLoginSecEvent>();
		}
	}

	/**
	 * Compare an instance of <code>EPPLoginSecData</code> with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLoginSecData)) {
			cat.error("EPPLoginSecData.equals(): " + aObject.getClass().getName() + " not EPPLoginSecData instance");

			return false;
		}

		EPPLoginSecData other = (EPPLoginSecData) aObject;

		// Events
		if (!EqualityUtil.equals(this.events, other.events)) {
			cat.error("EPPLoginSecData.equals(): events not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone an <code>EPPCodecComponent</code> instance.
	 *
	 * @return clone of concrete <code>EPPLoginSecData</code>
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {

		EPPLoginSecData clone = (EPPLoginSecData) super.clone();

		// Events
		clone.events = new ArrayList<EPPLoginSecEvent>();

		for (EPPLoginSecEvent event : this.events) {
			clone.events.add((EPPLoginSecEvent) event.clone());
		}

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>. 
	 */
	@Override
	public String getNamespace() {
		return EPPLoginSecExtFactory.NS;
	}

}
