/***********************************************************
Copyright (C) 2017 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.fee.v1_0;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Fee Check Extension that enables a client to pass a command along with
 * optionally the currency, period, and fee class to apply to the objects in the
 * check command.
 */
public class EPPFeeCheck implements EPPCodecComponent {

	/**
	 * XML local name for <code>EPPFeeCheck</code>.
	 */
	public static final String ELM_LOCALNAME = "check";

	/**
	 * XML root tag for <code>EPPFeeCheck</code>.
	 */
	public static final String ELM_NAME = EPPFeeExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPFeeCheck.class.getName(), EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for the currency
	 */
	private static final String ELM_CURRENCY = "currency";

	/**
	 * Commands to check for the fee
	 */
	private List<EPPFeeCommand> commands = new ArrayList<EPPFeeCommand>();

	/**
	 * OPTIONAL currency desired by the client.
	 */
	private String currency;

	/**
	 * Default constructor for <code>EPPFeeCheck</code>.
	 */
	public EPPFeeCheck() {
	}

	/**
	 * Constructor for <code>EPPFeeCheck</code> that takes a single command.
	 * 
	 * @param aCommand
	 *           Command to check
	 */
	public EPPFeeCheck(EPPFeeCommand aCommand) {
		this.addCommand(aCommand);
	}

	/**
	 * Constructor for <code>EPPFeeCheck</code> that takes a list of commands.
	 * 
	 * @param aCommands
	 *           List of commands to check
	 */
	public EPPFeeCheck(List<EPPFeeCommand> aCommands) {
		this.setCommands(aCommands);
	}

	/**
	 * Constructor for <code>EPPFeeCheck</code> that takes a single command
	 * attribute and the optional currency.
	 * 
	 * @param aCommand
	 *           Command to check
	 * @param aCurrency
	 *           OPTIONAL desired currency to return
	 */
	public EPPFeeCheck(EPPFeeCommand aCommand, String aCurrency) {
		this.addCommand(aCommand);
		this.setCurrency(aCurrency);
	}

	/**
	 * Constructor for <code>EPPFeeCheck</code> that takes a list of commands
	 * attribute and the optional currency.
	 * 
	 * @param aCommands
	 *           List of commands to check
	 * @param aCurrency
	 *           OPTIONAL desired currency to return
	 */
	public EPPFeeCheck(List<EPPFeeCommand> aCommands, String aCurrency) {
		this.setCommands(aCommands);
		this.setCurrency(aCurrency);
	}

	/**
	 * Are the commands defined?
	 * 
	 * @return <code>true</code> if the commands are defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasCommands() {
		return !this.commands.isEmpty();
	}

	/**
	 * Gets the list of commands if defined.
	 * 
	 * @return List of commands if defined; empty list otherwise.
	 */
	public List<EPPFeeCommand> getCommands() {
		return this.commands;
	}

	/**
	 * Adds a command to the list of commands.
	 * 
	 * @param aCommand
	 *           The command to add.
	 */
	public void addCommand(EPPFeeCommand aCommand) {
		if (aCommand == null)
			return;

		this.commands.add(aCommand);
	}

	/**
	 * Sets the list of commands.
	 * 
	 * @param aCommands
	 *           The commands to set.
	 */
	public void setCommands(List<EPPFeeCommand> aCommands) {
		if (aCommands == null)
			this.commands = new ArrayList<EPPFeeCommand>();
		else
			this.commands = aCommands;
	}

	/**
	 * Is the currency defined?
	 * 
	 * @return <code>true</code> if the currency is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasCurrency() {
		return (this.currency != null ? true : false);
	}

	/**
	 * Gets the currency value.
	 * 
	 * @return Currency if defined; <code>null</code> otherwise.
	 */
	public String getCurrency() {
		return this.currency;
	}

	/**
	 * Sets the currency value.
	 * 
	 * @param aCurrency
	 *           Currency value
	 */
	public void setCurrency(String aCurrency) {
		this.currency = aCurrency;
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 * 
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 * 
	 * @return Element Root element associated with the object
	 * 
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPFeeCheck</code>
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPFeeCheck.encode(Document)");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPFeeExtFactory.NS, ELM_NAME);

		// Currency
		EPPUtil.encodeString(aDocument, root, this.currency, EPPFeeExtFactory.NS,
		      EPPFeeExtFactory.NS_PREFIX + ":" + ELM_CURRENCY);

		// Commands
		EPPUtil.encodeCompList(aDocument, root, this.commands);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 * 
	 * @param aElement
	 *           <code>Element</code> to decode
	 * 
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// Currency
		this.currency = EPPUtil.decodeString(aElement, EPPFeeExtFactory.NS, ELM_CURRENCY);
		if (this.currency != null && this.currency.isEmpty()) {
			this.currency = null;
		}

		// Commands
		this.commands = EPPUtil.decodeCompList(aElement, EPPFeeExtFactory.NS, EPPFeeCommand.ELM_LOCALNAME,
		      EPPFeeCommand.class);
	}

	/**
	 * Clone an <code>EPPCodecComponent</code> instance.
	 * 
	 * @return clone of concrete <code>EPPFeeCheck</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {

		EPPFeeCheck clone = (EPPFeeCheck) super.clone();

		// Currency
		clone.currency = this.currency;

		// Commands
		clone.commands = new ArrayList<EPPFeeCommand>();
		for (EPPFeeCommand command : this.commands) {
			clone.commands.add((EPPFeeCommand) command.clone());
		}

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of <code>EPPFeeCheck</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPFeeCheck)) {
			cat.error("EPPFeeCheck.equals(): " + aObject.getClass().getName() + " not EPPFeeCheck instance");

			return false;
		}

		EPPFeeCheck other = (EPPFeeCheck) aObject;

		// Commands
		if (!EqualityUtil.equals(this.commands, other.commands)) {
			cat.error("EPPFeeCheck.equals(): commands not equal");
			return false;
		}

		// Currency
		if (!EqualityUtil.equals(this.currency, other.currency)) {
			cat.error("EPPFeeCheck.equals(): currency not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>. 
	 */
	@Override
	public String getNamespace() {
		return EPPFeeExtFactory.NS;
	}

}
