/**************************************************************************
 *                                                                        *
 * The information in this document is proprietary to VeriSign, Inc.      *
 * It may not be used, reproduced or disclosed without the written        *
 * approval of VeriSign.                                                  *
 *                                                                        *
 * VERISIGN PROPRIETARY & CONFIDENTIAL INFORMATION                        *
 *                                                                        *
 *                                                                        *
 * Copyright (c) 2011 VeriSign, Inc.  All rights reserved.                *
 *                                                                        *
 *************************************************************************/

package com.verisign.epp.codec.relateddomainext;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;

/**
 * EPPCodecComponent that encodes and decodes a <relDom:transfer> tag associated
 * with a Domain-Transfer command.
 * <p>
 * Title: EPP 1.0 Related Domain - Transfer tag
 * </p>
 * <p>
 * Description: The EPPRelatedDomainExtTransfer object represents the collection
 * of domains that must be transfered atomically. As such it is composed of a
 * collection of {@link EPPRelatedDomainExtDomain} objects. <br/>
 * As XML, it is represented by a <relDom:transfer> element containing a number
 * of <relDom:domain> elements.
 * </p>
 * <p>
 * Copyright: Copyright (c) 2013
 * </p>
 * <p>
 * Company: VeriSign
 * </p>
 * 
 * @author nchigurupati
 * @version 1.0
 */
public class EPPRelatedDomainExtTransfer implements EPPCodecComponent {

	/**
	 * Serial version id - increment this if the structure changes.
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(
			EPPRelatedDomainExtTransfer.class.getName(), EPPCatFactory
					.getInstance()
					.getFactory() );

	/**
	 * Element tag name for Transfer
	 */
	public static final String ELM_NAME = EPPRelatedDomainExtFactory.NS_PREFIX
			+ ":transfer";

	/**
	 * List of domain names represented by the {@link EPPRelatedDomainExtDomain}
	 * to be associated with the <relDom:transfer> element
	 */
	private List domains = null;


	/**
	 * Default constructor
	 */
	public EPPRelatedDomainExtTransfer () {
	}


	/**
	 * Constructor with a list of domains to be transferred.
	 * 
	 * @param aDomains
	 */
	public EPPRelatedDomainExtTransfer ( final List aDomains ) {
		this.domains = aDomains;

	}


	/**
	 * A deep clone of the EPPRelatedDomainExtTransfer
	 * 
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Object clone () throws CloneNotSupportedException {
		final EPPRelatedDomainExtTransfer theClone =
				new EPPRelatedDomainExtTransfer();
		theClone.domains = new ArrayList();

		if ( this.domains != null ) {
			for ( final Iterator iterator = this.domains.iterator(); iterator
					.hasNext(); ) {
				final Object attrObject = iterator.next();
				if ( attrObject != null ) {
					final EPPRelatedDomainExtDomain attr =
							(EPPRelatedDomainExtDomain) attrObject;
					theClone.domains.add( attr.clone() );
				}
			}
		}

		return theClone;
	}


	/**
	 * Populate the data of this instance with the data stored in the given
	 * Element of the DOM tree
	 * 
	 * @param aElement
	 *        The root element of the report fragment of XML
	 * @throws EPPDecodeException
	 *         Thrown if any errors occur during decoding.
	 */
	public void decode ( final Element aElement ) throws EPPDecodeException {

		this.domains =
				EPPUtil
						.decodeCompList( aElement, EPPRelatedDomainExtFactory.NS,
								EPPRelatedDomainExtDomain.ELM_NAME,
								EPPRelatedDomainExtDomain.class );
		try {
			validateState();
		}
		catch ( final EPPCodecException e ) {
			throw new EPPDecodeException(
					"Invalid state on EPPRelatedDomainExtTransfer.decode: " + e );
		}
	}


	/**
	 * Append all data from the list of the list of transfer related domains
	 * represented by {@link EPPRelatedDomainExtDomain} to given DOM Document
	 * 
	 * @param aDocument
	 *        The DOM Document to append data to
	 * @return Encoded DOM <code>Element</code>
	 * @throws EPPEncodeException
	 *         Thrown when errors occur during the encode attempt or if the
	 *         instance is invalid.
	 */
	public Element encode ( final Document aDocument ) throws EPPEncodeException {

		if ( aDocument == null ) {
			throw new EPPEncodeException( "aDocument is null"
					+ " in EPPRelatedDomainExtTransfer.encode(Document)" );
		}

		try {
			// Validate States
			validateState();
		}
		catch ( final EPPCodecException e ) {
			cat
					.error( "EPPRelatedDomainExtTransfer.encode(): Invalid state on encode: "
							+ e );
			throw new EPPEncodeException(
					"EPPRelatedDomainExtTransfer invalid state: " + e );
		}

		final Element root =
				aDocument.createElementNS( EPPRelatedDomainExtFactory.NS, ELM_NAME );
		EPPUtil.encodeCompList( aDocument, root, this.domains );
		return root;
	}


	/**
	 * A deep comparison of this with another EPPRelatedDomainExtTransfer.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals ( final Object aObj ) {

		if ( !(aObj instanceof EPPRelatedDomainExtTransfer) ) {
			return false;
		}

		final EPPRelatedDomainExtTransfer theComp =
				(EPPRelatedDomainExtTransfer) aObj;

		if ( !EPPUtil.equalLists( this.domains, theComp.domains ) ) {
			cat
					.error( "EPPRelatedDomainExtTransfer.equals(): TransferDomains not equal" );
			return false;
		}

		return true;
	}


	/**
	 * Returns the list of Transfer related domain names represented by
	 * {@link EPPRelatedDomainExtName}
	 * 
	 * @return the relatedDomains
	 */
	public List getDomains () {
		return this.domains;
	}


	/**
	 * Sets the list of transfer related domain names represented by
	 * {@link EPPRelatedDomainExtDomain}
	 * 
	 * @param aDomains
	 *        the relatedDomains to set
	 */
	public void setDomains ( final List aDomains ) {
		this.domains = aDomains;
	}


	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString () {
		return "EPPRelatedDomainExtTransfer [relatedDomains=" + this.domains + "]";
	}


	/**
	 * Validate the state of the <code>EPPRelatedDomainExtTransfer</code>
	 * instance. A valid state means that all of the required attributes have been
	 * set. If validateState returns without an exception, the state is valid. If
	 * the state is not valid, the <code>EPPCodecException</code> will contain a
	 * description of the error. throws EPPCodecException State error. This will
	 * contain the name of the attribute that is not valid.
	 * 
	 * @throws EPPCodecException
	 *         Thrown if the instance is in an invalid state
	 */
	private void validateState () throws EPPCodecException {
		if ( (this.domains == null) || (this.domains.size() == 0) ) {
			throw new EPPCodecException(
					"EPPRelatedDomainExtTransfer contains no  elements." );
		}

		final Iterator iterator = this.domains.iterator();
		while ( iterator.hasNext() ) {
			final EPPRelatedDomainExtDomain domain =
					(EPPRelatedDomainExtDomain) iterator.next();
			if ( domain == null ) {
				throw new EPPCodecException(
						"EPPRelatedDomainExtTransfer: contains null EPPRelatedDomainExtDomain  element." );
			}
			domain.validateStateForTransfer();
		}

	}

}
