/***********************************************************
 Copyright (C) 2012 VeriSign, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/

package com.verisign.epp.codec.signedMark;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.xml.crypto.MarshalException;
import javax.xml.crypto.XMLStructure;
import javax.xml.crypto.dom.DOMStructure;
import javax.xml.crypto.dsig.CanonicalizationMethod;
import javax.xml.crypto.dsig.DigestMethod;
import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.Transform;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.XMLSignatureFactory;
import javax.xml.crypto.dsig.dom.DOMSignContext;
import javax.xml.crypto.dsig.dom.DOMValidateContext;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfoFactory;
import javax.xml.crypto.dsig.keyinfo.X509Data;
import javax.xml.crypto.dsig.spec.C14NMethodParameterSpec;
import javax.xml.crypto.dsig.spec.TransformParameterSpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.codec.binary.Base64;
import org.apache.log4j.Logger;
import org.apache.xerces.dom.DocumentImpl;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.codec.mark.EPPMark;
import com.verisign.epp.exception.EPPException;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Class for the signed mark, which contains the mark ({@link EPPMark}), and
 * additional elements associated with the signing of the mark like the serial
 * number of the signed mark, the expiration of the signed mark, and the
 * <code>XMLSignature</code> itself.
 */
public class EPPSignedMark implements EPPCodecComponent {

	/** Namespace URI associated with EPPLaunchExtFactory. */
	public static final String NS = "urn:ietf:params:xml:ns:signedMark-1.0";

	/** Namespace prefix associated with EPPLaunchExtFactory. */
	public static final String NS_PREFIX = "signedMark";

	/** XML Schema definition for EPPLaunchExtFactory */
	public static final String NS_SCHEMA = "urn:ietf:params:xml:ns:signedMark-1.0 signedMark-1.0.xsd";

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPSignedMark.class.getName(),
			EPPCatFactory.getInstance().getFactory());

	/**
	 * Constant for the mark local name for signedMark element
	 */
	public static final String ELM_SIGNED_MARK_LOCALNAME = "signedMark";

	/**
	 * Constant for the mark tag for signedMark element
	 */
	public static final String ELM_SIGNED_MARK_NAME = NS_PREFIX + ":"
			+ ELM_SIGNED_MARK_LOCALNAME;

	/**
	 * Constant for the mark local name for signedMark element
	 */
	public static final String ELM_ENCODED_SIGNED_MARK_LOCALNAME = "encodedSignedMark";

	/**
	 * Constant for the mark tag for signedMark element
	 */
	public static final String ELM_ENCODED_SIGNED_MARK_NAME = NS_PREFIX + ":"
			+ ELM_ENCODED_SIGNED_MARK_LOCALNAME;

	/**
	 * Element local name for serial number for the signed mark
	 */
	private static final String ELM_ID = "id";

	/**
	 * Element local name for the date the signed mark was created
	 */
	private static final String ELM_NOT_BEFORE = "notBefore";

	/**
	 * Element local name for the date of expiration of the signed mark
	 */
	private static final String ELM_NOT_AFTER = "notAfter";

	/**
	 * The ID attribute name
	 */
	private static final String ATTR_ID = "id";

	/**
	 * The ID attribute value used for the signed mark
	 */
	private static final String ATTR_ID_VALUE = "signedMark";

	/**
	 * XML local name for the root element of the signed mark. This should be
	 * set to either <code>ELM_SIGNED_MARK_LOCALNAME</code> or
	 * <code>ELM_ENCODED_SIGNED_MARK_LOCALNAME</code>.
	 */
	private String localName = ELM_SIGNED_MARK_LOCALNAME;

	/**
	 * Identifier of the signed mark
	 */
	private String id;

	/**
	 * Issuer of the signed mark.
	 */
	private EPPIssuer issuer;

	/**
	 * the date of creation of the signed mark
	 */
	private Date notBefore;

	/**
	 * the date of expiration of the signed mark
	 */
	private Date notAfter;

	/**
	 * Mark associated with the signed mark
	 */
	private EPPMark mark;

	/**
	 * XML Signature of the &lt;signedMark&gt; element.
	 */
	private transient XMLSignature signature;

	/**
	 * DOM <code>Element</code> representing the <code>signature</code>.
	 */
	private Element signatureElement;

	/**
	 * XML Signature Factory
	 */
	private static XMLSignatureFactory sigFactory = XMLSignatureFactory
			.getInstance("DOM");

	/**
	 * Create an <code>EPPSignedMark</code> instance. Use the setter methods to
	 * set the attributes of the instance.
	 */
	public EPPSignedMark() {
	}

	/**
	 * Create an <code>EPPSignedMark</code> with the id, issuer, not before
	 * date, not after date, and the mark attributes of the signed mark. The
	 * default encoding is XML and the signature must be generated by calling
	 * {@link #sign(PrivateKey)}.
	 * 
	 * @param aId
	 *            Identifier of signed mark
	 * @param aIssuer
	 *            Signed mark issuer information
	 * @param aNotBefore
	 *            Date and time that the signed mark was created.
	 * @param aNotAfter
	 *            Date and time that the signed mark expires.
	 * @param aMark
	 *            Mark information
	 */
	public EPPSignedMark(String aId, EPPIssuer aIssuer, Date aNotBefore,
			Date aNotAfter, EPPMark aMark) {
		this.id = aId;
		this.issuer = aIssuer;
		this.notBefore = aNotBefore;
		this.notAfter = aNotAfter;
		this.mark = aMark;
	}

	/**
	 * Create an <code>EPPSignedMark</code> will all of the attributes except
	 * for the signature that must be generated by calling
	 * {@link #sign(PrivateKey)}.
	 * 
	 * @param aId
	 *            Identifier of signed mark
	 * @param aIssuer
	 *            Signed mark issuer information
	 * @param aNotBefore
	 *            Date and time that the signed mark was created.
	 * @param aNotAfter
	 *            OPTIONAL date and time that the mark expires. If set to
	 *            <code>null</code> there is no expiration.
	 * @param aMark
	 *            Mark information
	 * @param aBase64Encoded
	 *            Base64 encode the signed mark by enclosing in the
	 *            &lt;encodedSignedMark&gt; element
	 */
	public EPPSignedMark(String aId, EPPIssuer aIssuer, Date aNotBefore,
			Date aNotAfter, EPPMark aMark, boolean aBase64Encoded) {
		this.id = aId;
		this.issuer = aIssuer;
		this.notBefore = aNotBefore;
		this.notAfter = aNotAfter;
		this.mark = aMark;
		this.setBase64Encode(aBase64Encoded);
	}

	/**
	 * Clone <code>EPPSignedMark</code>.
	 * 
	 * @return clone of <code>EPPSignedMark</code>
	 * @exception CloneNotSupportedException
	 *                standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPSignedMark clone = (EPPSignedMark) super.clone();

		// Issuer
		if (this.issuer != null) {
			clone.issuer = (EPPIssuer) this.issuer.clone();
		}

		// Mark
		if (this.mark != null) {
			clone.mark = (EPPMark) this.mark.clone();
		}

		return clone;
	}

	/**
	 * Digitally sign the signed mark using the passed private key. The
	 * resulting signature is stored as an attribute. The signature can be
	 * retrieved with the {@link #getSignature()} method. No certificates will
	 * be added using this method. If certificates need to be added use
	 * {@link #sign(PrivateKey, Certificate[])}.
	 * 
	 * @param aPrivateKey
	 *            Private key used to sign the signed mark
	 * 
	 * @throws EPPException
	 *             Error creating the digital signature
	 */
	public void sign(PrivateKey aPrivateKey) throws EPPException {
		cat.debug("EPPSignedMark.sign(PrivateKey): enter");
		this.sign(aPrivateKey, null);
		cat.debug("EPPSignedMark.sign(PrivateKey): exit");
	}

	/**
	 * Digitally sign the signed mark using the passed private key and a chain
	 * of certificates. The resulting signature is stored as an attribute. The
	 * signature can be retrieved with the {@link #getSignature()} method.
	 * 
	 * @param aPrivateKey
	 *            Private key used to sign the signed mark
	 * @param aCertChain
	 *            Certificate chain to include in the XMLSignature associated
	 *            with the private key. Pass <code>null</code> to not include
	 *            the certificate chain in the XMLSignature.
	 * 
	 * @throws EPPException
	 *             Error creating the digital signature
	 */
	public void sign(PrivateKey aPrivateKey, Certificate[] aCertChain)
			throws EPPException {
		cat.debug("EPPSignedMark.sign(PrivateKey, Certificate[]): enter");

		// Required parameter is null?
		if (aPrivateKey == null) {
			throw new EPPException(
					"EPPSignedMark.sign(PrivateKey, Certificate[]): null aPrivateKey parameter");
		}

		Document document = new DocumentImpl();

		boolean origBase64Encoded = this.isBase64Encode();
		this.setBase64Encode(false);

		Element root = this.encode(document);
		document.appendChild(root);

		this.setBase64Encode(origBase64Encoded);

		try {
			DigestMethod digestMethod = sigFactory.newDigestMethod(
					DigestMethod.SHA256, null);

			List transforms = new ArrayList<Transform>();
			transforms.add(sigFactory.newTransform(Transform.ENVELOPED,
					(TransformParameterSpec) null));
			Reference xmlSigRef = sigFactory.newReference("#" + ATTR_ID_VALUE,
					digestMethod, transforms, null, null);

			// Create the SignedInfo
			SignedInfo signedInfo = sigFactory
					.newSignedInfo(
							sigFactory
									.newCanonicalizationMethod(
											CanonicalizationMethod.INCLUSIVE_WITH_COMMENTS,
											(C14NMethodParameterSpec) null),
							sigFactory
									.newSignatureMethod(
											"http://www.w3.org/2001/04/xmldsig-more#rsa-sha256",
											null), Collections
									.singletonList(xmlSigRef));

			// Add certificate chain to signature?
			KeyInfo keyInfo = null;
			if (aCertChain != null) {
				cat.debug("EPPSignedMark.sign(PrivateKey, Certificate[]): certificate chain passed");

				KeyInfoFactory keyInfoFactory = sigFactory.getKeyInfoFactory();
				List<X509Certificate> certChain = new ArrayList<X509Certificate>();
				for (Certificate cert : aCertChain) {
					if (cert == null || !(cert instanceof X509Certificate)) {
						throw new EPPException(
								"EPPSignedMark.sign(PrivateKey, Certificate[]): Null or invalid certificate type");
					}
					certChain.add((X509Certificate) cert);
					cat.debug("EPPSignedMark.sign(PrivateKey, Certificate[]): Added certificate ["
							+ cert + "] to X509Certificate list");
				}
				List<X509Data> certDataList = new ArrayList<X509Data>();
				certDataList.add(keyInfoFactory.newX509Data(certChain));
				keyInfo = keyInfoFactory.newKeyInfo(certDataList);
			}
			else {
				cat.debug("EPPSignedMark.sign(PrivateKey, Certificate[]): null certificate chain passed, no certificates added");
			}

			DOMSignContext signContext = new DOMSignContext(aPrivateKey, root);
			signContext.setDefaultNamespacePrefix("dsig");
			this.signature = sigFactory.newXMLSignature(signedInfo, keyInfo);
			this.signature.sign(signContext);

			this.signatureElement = EPPUtil.getElementByTagNameNS(root,
					XMLSignature.XMLNS, "Signature");
		}
		catch (Exception ex) {
			ex.printStackTrace();
			cat.error("Error signing the EPPSignedMark: " + ex);
			throw new EPPException(
					"EPPSignedMark.sign(PrivateKey, Certificate[]): Error signing the EPPSignedMark");
		}

		cat.debug("EPPSignedMark.sign(PrivateKey, Certificate[]): exit");
	}

	/**
	 * Validate the signature attribute against the signed mark attributes by
	 * using the public key of the certificate or the top certificate in the
	 * certificate chain contained in the <code>XMLSignature</code> with using
	 * the passed PKIX parameters to the PKIX <code>CertPathValidator</code>
	 * algorithm. The trust store can be loaded and used to create an instance
	 * of <code>PKIXParameters</code> to verify the certificate chain included
	 * in the <code>XMLSignature</code> with the trust anchors included in the
	 * trust store. This method will automatically synchronize the
	 * <code>aPKIXParameters</code> parameter when used, since it is not
	 * thread-safe. Use {@link #validate(PKIXParameters, boolean)} to explicitly
	 * set the <code>aPKIXParameters</code> synchronization setting.
	 * 
	 * @param aPKIXParameters
	 *            Parameters used as input for the PKIX
	 *            <code>CertPathValidator</code> algorithm.
	 * 
	 * @return <code>true</code> if valid; <code>false</code> otherwise.
	 */
	public boolean validate(PKIXParameters aPKIXParameters) {
		return this.validate(aPKIXParameters, true);
	}

	/**
	 * Validate the signature attribute against the signed mark attributes by
	 * using the public key of the certificate or the top certificate in the
	 * certificate chain contained in the <code>XMLSignature</code> with using
	 * the passed PKIX parameters to the PKIX <code>CertPathValidator</code>
	 * algorithm. The trust store can be loaded and used to create an instance
	 * of <code>PKIXParameters</code> to verify the certificate chain included
	 * in the <code>XMLSignature</code> with the trust anchors included in the
	 * trust store.
	 * 
	 * @param aPKIXParameters
	 *            Parameters used as input for the PKIX
	 *            <code>CertPathValidator</code> algorithm.
	 * @param aSynchronizePKIXParameters
	 *            Should the <code>aPKIXParameters</code> be synchronized inside
	 *            the method? If there is no reason to synchronize, then
	 *            <code>false</code> can be passed to increase performance.
	 * 
	 * @return <code>true</code> if valid; <code>false</code> otherwise.
	 */
	public boolean validate(PKIXParameters aPKIXParameters,
			boolean aSynchronizePKIXParameters) {

		cat.debug("validate(PKIXParameters): enter");

		boolean valid = false;

		try {
			Element sigElm = this.findSignatureElement();

			DOMStructure domStructure = new DOMStructure(sigElm);

			this.signature = sigFactory.unmarshalXMLSignature(domStructure);

			// No key info found?
			if (this.signature.getKeyInfo() == null) {
				throw new Exception("No key info found in Signature");
			}

			List<X509Certificate> certificates = null;

			List<XMLStructure> keyContent = this.signature.getKeyInfo()
					.getContent();

			// For each signature keyInfo item
			for (XMLStructure currInfo : keyContent) {

				// X509 data?
				if (currInfo instanceof X509Data) {

					List x509Data = ((X509Data) currInfo).getContent();

					if (x509Data == null) {
						continue;
					}

					// For each X509Data element
					for (Object currX509Data : x509Data) {

						// X509Certificate?
						if (currX509Data instanceof X509Certificate) {

							if (certificates == null) {
								certificates = new ArrayList<X509Certificate>();
							}

							X509Certificate x509Cert = (X509Certificate) currX509Data;

							// Check validity of certificate
							x509Cert.checkValidity();

							cat.debug("validate(PKIXParameters): Found X509Certificate ["
									+ x509Cert + "]");
							certificates.add(x509Cert);
						}
					}
				}

				// No Certificates found?
				if (certificates == null || certificates.isEmpty()) {
					throw new Exception("No certificates found in Signature");
				}

				CertificateFactory certFactory = CertificateFactory
						.getInstance("X.509");
				CertPath certPath = certFactory.generateCertPath(certificates);

				// Validate certificate path against trust anchors
				// (aPKIXParameters)
				CertPathValidator pathValidator = CertPathValidator
						.getInstance("PKIX");

				// Must synchronize around the use of PKIXParameters
				// since it is NOT thread-safe.
				if (aSynchronizePKIXParameters) {
					synchronized (aPKIXParameters) {
						pathValidator.validate(certPath, aPKIXParameters);
					}
				}
				else {
					pathValidator.validate(certPath, aPKIXParameters);
				}
			}

			// Get the public key from top certificate to validate signature
			X509Certificate cert = certificates.get(0);
			cat.debug("validate(PKIXParameters): Getting public key from top certificate ["
					+ cert + "]");
			PublicKey publicKey = cert.getPublicKey();

			valid = this.validate(sigElm, publicKey);
		}
		catch (Exception ex) {
			cat.error("validate(PKIXParameters): Error validating the EPPSignedMark: "
					+ ex);
			valid = false;
		}

		cat.debug("validate(PKIXParameters): exit, valid = " + valid);
		return valid;

	}

	/**
	 * Validate the signature attribute against the signed mark attributes.
	 * 
	 * @param aPublicKey
	 *            Public used to validate the signature
	 * @return <code>true</code> if valid; <code>false</code> otherwise.
	 */
	public boolean validate(PublicKey aPublicKey) {
		cat.debug("validate(PublicKey): enter");

		boolean valid = false;

		try {
			Element sigElm = this.findSignatureElement();
			valid = this.validate(sigElm, aPublicKey);
		}
		catch (Exception ex) {
			cat.error("validate(PublicKey): Error validating the EPPSignedMark: "
					+ ex);
			valid = false;
		}

		cat.debug("validate(PublicKey): exit, valid = " + valid);
		return valid;
	}

	/**
	 * Encode the signed mark to a <code>byte[]</code>.
	 * 
	 * @return Encoded signed mark
	 * @throws EPPEncodeException
	 *             Error encoding the signed mark
	 */
	public byte[] encode() throws EPPEncodeException {
		cat.debug("EPPSignedMark.encode(): enter");

		ByteArrayOutputStream os = new ByteArrayOutputStream();
		try {
			Document doc = new DocumentImpl();
			Element root = this.encode(doc);
			doc.appendChild(root);

			TransformerFactory transFac = TransformerFactory.newInstance();
			Transformer trans = transFac.newTransformer();

			trans.transform(new DOMSource(root), new StreamResult(os));
		}
		catch (EPPEncodeException ex) {
			throw ex;
		}
		catch (Exception ex) {
			cat.error("Error encoding signed mark to byte[]: " + ex);
			throw new EPPEncodeException("Error encoding signed mark to byte[]");
		}

		cat.debug("EPPSignedMark.encode(): exit");
		return os.toByteArray();
	}

	/**
	 * Sets all this instance's data in the given XML document
	 * 
	 * @param aDocument
	 *            a DOM Document to attach data to.
	 * @return The root element of this component.
	 * 
	 * @throws EPPEncodeException
	 *             Thrown if any errors prevent encoding.
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {
		cat.debug("EPPSignedMark.encode(Document): enter");

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null"
					+ " on in EPPSignedMark.encode(Document)");
		}

		Element root = aDocument.createElementNS(NS, NS_PREFIX + ":"
				+ this.localName);

		// Handle Base64 encoded signed mark
		if (this.localName.equals(ELM_ENCODED_SIGNED_MARK_LOCALNAME)) {
			this.localName = ELM_SIGNED_MARK_LOCALNAME;

			byte[] signedMarkXml = this.encode();

			this.localName = ELM_ENCODED_SIGNED_MARK_LOCALNAME;

			String base64EncodedText = new String(Base64.encodeBase64(
					signedMarkXml, true));

			Text currVal = aDocument.createTextNode(base64EncodedText);
			root.appendChild(currVal);
			cat.debug("EPPSignedMark.encode(Document): exit - encoded");
			return root;
		}
		
		// Validate required attributes
		if (this.id == null) {
			throw new EPPEncodeException("Signed mark id is required.");
		}
		if (this.issuer == null) {
			throw new EPPEncodeException("Signed mark issuer is required.");
		}
		if (this.notBefore == null) {
			throw new EPPEncodeException("Signed mark notBefore is required.");
		}
		if (this.notAfter == null) {
			throw new EPPEncodeException("Signed mark notAfter is required.");
		}
		if (this.mark == null) {
			throw new EPPEncodeException("Signed mark mark is required.");
		}

		// Add the "id" Attribute
		root.setAttribute(ATTR_ID, ATTR_ID_VALUE);
		root.setIdAttribute(ATTR_ID, true);

		// Id
		EPPUtil.encodeString(aDocument, root, this.id, NS, NS_PREFIX + ":"
				+ ELM_ID);

		// Issuer
		EPPUtil.encodeComp(aDocument, root, this.issuer);

		// Not Before
		EPPUtil.encodeTimeInstant(aDocument, root, this.notBefore, NS,
				NS_PREFIX + ":" + ELM_NOT_BEFORE);

		// Not After
		EPPUtil.encodeTimeInstant(aDocument, root, this.notAfter, NS, NS_PREFIX
				+ ":" + ELM_NOT_AFTER);

		// Mark
		EPPUtil.encodeComp(aDocument, root, this.mark);

		// Signature
		if (this.signatureElement != null) {
			this.signatureElement = (Element) aDocument.importNode(
					this.signatureElement, true);
			root.appendChild(this.signatureElement);
		}

		cat.debug("EPPSignedMark.encode(Document): exit - normal");
		return root;
	}

	/**
	 * Decode the <code>EPPSignedMark</code> attributes from the input
	 * <code>byte[]</code>.
	 * 
	 * @param aSignedMarkArray
	 *            <code>byte[]</code> to decode the attribute values
	 * @throws EPPDecodeException
	 *             Error decoding the <code>byte[]</code>.
	 */
	public void decode(byte[] aSignedMarkArray) throws EPPDecodeException {
		cat.debug("EPPSignedMark.decode(byte[]): enter");

		try {
			ByteArrayInputStream is = new ByteArrayInputStream(aSignedMarkArray);
			DocumentBuilderFactory docFactory = DocumentBuilderFactory
					.newInstance();
			docFactory.setNamespaceAware(true);
			DocumentBuilder docBuilder;
			docBuilder = docFactory.newDocumentBuilder();
			Document doc = docBuilder.parse(is);

			this.decode(doc.getDocumentElement());
		}
		catch (Exception ex) {
			throw new EPPDecodeException("Error decoding signed mark array: "
					+ ex);
		}

		cat.debug("EPPSignedMark.decode(byte[]): exit");
	}

	/**
	 * Decode the <code>EPPSignedMark</code> component
	 * 
	 * @param aElement
	 *            Root element of the <code>EPPSignedMark</code>
	 * @throws EPPDecodeException
	 *             Error decoding the <code>EPPSignedMark</code>
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		cat.debug("EPPSignedMark.decode(Element): enter");

		/**
		 * Sets the local name of the root element
		 */
		this.localName = aElement.getLocalName();

		// Base64 encoded element
		if (this.localName.equals(ELM_ENCODED_SIGNED_MARK_LOCALNAME)) {

			String base64SignedMark = EPPUtil.getTextContent(aElement);

			byte[] signedMarkXML = Base64.decodeBase64(base64SignedMark);

			this.localName = ELM_SIGNED_MARK_LOCALNAME;
			this.decode(signedMarkXML);
			this.localName = ELM_ENCODED_SIGNED_MARK_LOCALNAME;
			cat.debug("EPPSignedMark.decode(Element): exit - encoded");
			return;
		}

		// Id
		this.id = EPPUtil.decodeString(aElement, NS, ELM_ID);

		// Issuer
		this.issuer = (EPPIssuer) EPPUtil.decodeComp(aElement,
				EPPSignedMark.NS, EPPIssuer.ELM_LOCALNAME, EPPIssuer.class);

		// Not Before
		this.notBefore = EPPUtil
				.decodeTimeInstant(aElement, NS, ELM_NOT_BEFORE);

		// Not After
		this.notAfter = EPPUtil.decodeTimeInstant(aElement, NS, ELM_NOT_AFTER);

		// Mark
		this.mark = (EPPMark) EPPUtil.decodeComp(aElement, EPPMark.NS,
				EPPMark.ELM_LOCALNAME, EPPMark.class);

		// Signature
		this.signatureElement = EPPUtil.getElementByTagNameNS(aElement,
				XMLSignature.XMLNS, "Signature");
		if (signatureElement != null) {
			DOMStructure structure = new DOMStructure(signatureElement);
			try {
				this.signature = sigFactory.unmarshalXMLSignature(structure);
			}
			catch (MarshalException ex) {
				throw new EPPDecodeException(
						"Error decoding the XML Signature: " + ex);
			}
		}

		cat.debug("EPPSignedMark.decode(Element): exit - normal");
	}

	/**
	 * implements a deep <code>EPPSignedMark</code> compare.
	 * 
	 * @param aObject
	 *            <code>EPPSignedMark</code> instance to compare with
	 * 
	 * @return true if equal false otherwise
	 */
	public boolean equals(Object aObject) {

		if (!(aObject instanceof EPPSignedMark)) {
			cat.error("EPPSignedMark.equals(): aObject is not an EPPSignedMark");
			return false;
		}

		EPPSignedMark other = (EPPSignedMark) aObject;

		// Id
		if (!EqualityUtil.equals(this.id, other.id)) {
			cat.error("EPPSignedMark.equals(): id not equal");
			return false;
		}
		
		// Issuer
		if (!EqualityUtil.equals(this.issuer, other.issuer)) {
			cat.error("EPPSignedMark.equals(): issuer not equal");
			return false;
		}		

		// Not Before
		if (!EqualityUtil.equals(this.notBefore, other.notBefore)) {
			cat.error("EPPSignedMark.equals(): notBefore not equal");
			return false;
		}

		// Not After
		if (!EqualityUtil.equals(this.notAfter, other.notAfter)) {
			cat.error("EPPSignedMark.equals(): notAfter not equal");
			return false;
		}

		// Mark
		if (!EqualityUtil.equals(this.mark, other.mark)) {
			cat.error("EPPSignedMark.equals(): mark not equal");
			return false;
		}

		return true;
	}

	/**
	 * Gets the XML local name for the signed mark.
	 * 
	 * @return Either <code>ELM_SIGNED_MARK_LOCALNAME</code> or
	 *         <code>ELM_ENCODED_SIGNED_MARK_LOCALNAME</code>
	 */
	public String getLocalName() {
		return this.localName;
	}

	/**
	 * Sets the XML local name for the signed mark.
	 * 
	 * @param aLocalName
	 *            Either <code>ELM_SIGNED_MARK_LOCALNAME</code> or
	 *            <code>ELM_ENCODED_SIGNED_MARK_LOCALNAME</code>
	 */
	public void setLocalName(String aLocalName) {
		this.localName = aLocalName;
	}

	/**
	 * Gets the identifier of the signed mark.
	 * 
	 * @return The identifier for the signed mark if set; <code>null</code>
	 *         otherwise.
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * Sets the identifier of the signed mark.
	 * 
	 * @param aId
	 *            Identifier of the signed mark.
	 */
	public void setId(String aId) {
		this.id = aId;
	}
	
	/**
	 * Gets issuer of the signed mark.
	 * 
	 * @return The issuer of the signed mark if defined:
	 *         <code>null</code> otherwise.
	 */
	public EPPIssuer getIssuer() {
		return this.issuer;
	}

	/**
	 * Sets the issuer of the signed mark.
	 * 
	 * @param aIssuer
	 *            Issuer of the signed mark.
	 */
	public void setIssuer(EPPIssuer aIssuer) {
		this.issuer = aIssuer;
	}
	

	/**
	 * Gets the date of creation of the signed mark.
	 * 
	 * @return the date of creation of the signed mark if set; <code>null</code>
	 *         otherwise.
	 */
	public Date getNotBefore() {
		return this.notBefore;
	}

	/**
	 * Sets the date of creation of the signed mark.
	 * 
	 * @param aNotBefore
	 *            The date of creation of the signed mark
	 */
	public void setNotBefore(Date aNotBefore) {
		this.notBefore = aNotBefore;
	}

	/**
	 * Gets the date of expiration of the signed mark.
	 * 
	 * @return the date of expiration of the signed mark if set;
	 *         <code>null</code> otherwise.
	 */
	public Date getNotAfter() {
		return this.notAfter;
	}

	/**
	 * Sets the date of expiration of the signed mark.
	 * 
	 * @param aNotAfter
	 *            The date of expiration of the signed mark
	 */
	public void setNotAfter(Date aNotAfter) {
		this.notAfter = aNotAfter;
	}

	/**
	 * Gets the mark associated with the signed mark.
	 * 
	 * @return The mark associated with the signed mark if defined:
	 *         <code>null</code> otherwise.
	 */
	public EPPMark getMark() {
		return this.mark;
	}

	/**
	 * Sets the mark associated with the signed mark.
	 * 
	 * @param aMark
	 *            Mark associated with the signed mark.
	 */
	public void setMark(EPPMark aMark) {
		this.mark = aMark;
	}

	/**
	 * Gets the <code>XMLSignature</code> associated with the signed mark.
	 * 
	 * @return <code>XMLSignature</code> instance if set; <code>null</code>
	 *         otherwise.
	 */
	public XMLSignature getSignature() {
		return this.signature;
	}

	/**
	 * The DOM <code>Element</code> of the <code>XMLSignature</code>.
	 * 
	 * @return The DOM <code>Element</code> of the <code>XMLSignature</code> if
	 *         set; <code>null</code> otherwise.
	 */
	public Element getSignatureElement() {
		return this.signatureElement;
	}

	/**
	 * Gets if the signed mark should be encoded in Base64 with the
	 * &lt;encodedSignedMark&gt; root element. The default value is
	 * <code>false</code>.
	 * 
	 * @return <code>true</code> if Base64 encode; <code>false</code> otherwise.
	 */
	public boolean isBase64Encode() {
		if (this.localName.equals(ELM_ENCODED_SIGNED_MARK_LOCALNAME)) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Sets if the signed mark should be encoded in Base64 with the
	 * &lt;encodedSignedMark&gt; root element.
	 * 
	 * @param aBase64Encode
	 *            <code>true</code> to Base64 encode; <code>false</code>
	 *            otherwise.
	 */
	public void setBase64Encode(boolean aBase64Encode) {
		if (aBase64Encode) {
			this.localName = ELM_ENCODED_SIGNED_MARK_LOCALNAME;
		}
		else {
			this.localName = ELM_SIGNED_MARK_LOCALNAME;
		}
	}

	/**
	 * Find the DOM <code>Signature Element</code>.
	 * 
	 * @return DOM <code>Signature Element</code>
	 * 
	 * @throws EPPException
	 *             Error finding DOM <code>Signature Element</code>
	 */
	private Element findSignatureElement() throws EPPException {
		cat.debug("EPPSignedMark.findSignatureElement(): exit");

		Element sigElm = null;

		try {
			Document document = new DocumentImpl();

			boolean origBase64Encoded = this.isBase64Encode();
			this.setBase64Encode(false);

			Element root = this.encode(document);
			document.appendChild(root);

			this.setBase64Encode(origBase64Encoded);

			sigElm = EPPUtil.getElementByTagNameNS(root, XMLSignature.XMLNS,
					"Signature");
		}
		catch (Exception ex) {
			cat.error("EPPSignedMark.findSignatureElement(): During processing to find Signature element");
			throw new EPPException(
					"Cannot find Signature element during processing");

		}

		if (sigElm == null) {
			cat.error("EPPSignedMark.findSignatureElement(): Cannot find Signature element");
			throw new EPPException("Cannot find Signature element");
		}

		cat.debug("EPPSignedMark.findSignatureElement(): exit");
		return sigElm;
	}

	/**
	 * Validate the signature attribute against the signed mark attributes given
	 * the <code>Signature</code> DOM <code>Element</code>.
	 * 
	 * @param aSigElm
	 *            DOM <code>Signature Element</code>
	 * @param aPublicKey
	 *            Public used to validate the signature
	 * @return <code>true</code> if valid; <code>false</code> otherwise.
	 */
	private boolean validate(Element aSigElm, PublicKey aPublicKey) {
		cat.debug("validate(Element, PublicKey): enter");

		boolean valid = false;

		try {
			DOMValidateContext valContext = new DOMValidateContext(aPublicKey,
					aSigElm);
			this.signature = sigFactory.unmarshalXMLSignature(valContext);

			if (this.signature.validate(valContext)) {
				valid = true;
			}
			else {
				valid = false;
				cat.error("validate(Element, PublicKey): validation status = "
						+ this.signature.getSignatureValue().validate(
								valContext));
				Iterator<?> i = this.signature.getSignedInfo().getReferences()
						.iterator();

				for (int j = 0; i.hasNext(); j++) {
					cat.error("validate(Element, PublicKey): ref[" + j
							+ "] validity status = "
							+ ((Reference) i.next()).validate(valContext));
				}

			}
		}
		catch (Exception ex) {
			cat.error("validate(Element, PublicKey): Error validating the EPPSignedMark: "
					+ ex);
			valid = false;
		}

		cat.debug("validate(Element, PublicKey): exit, valid = " + valid);
		return valid;
	}

}